(ns circleci.statsd
  (:require [clojure.tools.logging :refer (infof)]
            clj-statsd))

(defn setup
  [host port & opts]
  (apply clj-statsd/setup host port opts))

(defn increment
  ([k]   (clj-statsd/increment k 1 1.0 []))
  ([k v] (clj-statsd/increment k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/increment k v rate (map name tags))))

(defn decrement
  ([k]   (clj-statsd/decrement k 1 1.0 []))
  ([k v] (clj-statsd/decrement k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/decrement k v rate (map name tags))))

(defn timing
  ([k v] (clj-statsd/timing k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/timing k v rate (map name tags))))

(defn gauge
  ([k v] (clj-statsd/gauge k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/gauge k v rate (map name tags))))

(defn unique
  ([k v] (clj-statsd/unique k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/unique k v rate (map name tags))))

(defn- remove-custom-tags
  [tags]
  (remove (partial re-find #"^[Xx]-") tags))

(defn with-timing-metric-call
  "Wrap a call to f with timing instrumentation.
  'metric' is a keyword naming the metrics.
  'tags' is a vector of strings to tag the metric with.

  Tags that are prefixed with X- (e.g. X-foo-bar) are removed before submitting
  the timing metric to statsd.
  The prefixed tags and timing data are logged, allowing for analysis via logs.

  Use this for tags that are not appropriate for statsd services such as
  Datadog, e.g. tags with no bounds on the number of values."
  [metric tags f]
  (let [metric-tags (remove-custom-tags tags)
        start-time (System/nanoTime)]
    (try
      (f)
      (finally
        (let [elapsed-time (-> (System/nanoTime)
                               (- start-time)
                               (/ 1e6)
                               (#(Math/round %)))]
          (when (> (count tags) (count metric-tags))
            (infof "with-timing-metric %s %s: took %d milliseconds"
                   metric
                   tags
                   elapsed-time))
          (timing metric elapsed-time {:tags metric-tags}))))))

(defmacro with-timing-metric
  "Helper macro for calling with-timing-metric-call"
  [metric tags & body]
  `(with-timing-metric-call ~metric ~tags (fn [] ~@body)))
