(ns circle-vault.core
  (:require [clojure.edn :as edn]
            [clojure.set :as set]
            [clojure.string :as string]

            [cheshire.core :as json]
            [clj-http.client :as http]
            [clj-time.core :as time]
            [com.stuartsierra.component :as component]))

(def api-version "v1")

(defn- role-id []
  (System/getenv "VAULT_ROLE_ID"))

(defn- security-id []
  (System/getenv "VAULT_SECURITY_ID"))

(defn- valid-path?
  [path]
  (and (string? path)
       (not (string/blank? path))))

(defmulti auth!
  (fn [url method data] method))

(defmethod auth! :default
  [url method data]
  (throw (ex-info (format "Authentication method %s not implemented" method) {})))

(defmethod auth! :app-id
  [url method {:keys [app-id user-id]}]
  {:pre [(string? app-id), (string? user-id)]}
  (try
    (let [response (http/post (str url "/v1/auth/app-id/login")
                              {:content-type :json
                               :body (json/generate-string {:app_id app-id, :user_id user-id})})
          body (json/parse-string (:body response) true)]
      (get-in body [:auth :client_token]))
    (catch Exception e
      :error)))

(defmethod auth! :app-role
  [url method {:keys [role-id secret-id]}]
  {:pre [(string? role-id) (string? secret-id)]}
  (let [response (http/post (str url "/v1/auth/approle/login")
                            {:content-type :json
                             :body (json/generate-string {:role_id role-id
                                                          :secret_id secret-id})})
        body (json/parse-string (:body response) true)]
    (get-in body [:auth :client_token])))

(defn authenticate [vault-client]
  (auth! (:api-url vault-client)
         (:auth-method vault-client)
         (:auth-data vault-client)))


(defrecord VaultClient [api-url auth-method auth-data token]
  component/Lifecycle

  (start [this]
    (if-let [token (authenticate this)]
      (assoc this :token token)
      ;; TODO: what can we do with the component when this doesn't work?
      this))

  (stop [this]
    (assoc this :token nil)))

(defn new-vault-client
  [api-url auth-method auth-data]
  (map->VaultClient {:api-url api-url
                     :auth-method auth-method
                     :auth-data auth-data
                     :token nil}))

(defn vault-client
  [url role-id secret-id]
  (-> (new-vault-client url :app-role {:role-id role-id :secret-id secret-id}) (component/start)))

(defn- wrap-request
  [client path method data]
  {:pre [(valid-path? path)]}

  (let [api-base-path (str "/" api-version "/")
        request-path (str (:api-url client) api-base-path path)
        token {"x-vault-token" (:token client)}]
    (case method
      :get (http/get request-path
                     {:headers token
                      :throw-exceptions false})
      :put (http/put request-path
                     {:headers token
                      :throw-exceptions false
                      :content-type :json
                      :body (json/generate-string data)}))))

(defn- wrap-response
  [response]
  ;; This "mangles" the data in that every JSON key string is
  ;; keywordised
  (let [body (json/parse-string (:body response) true)]
    {:status (:status response)
     :lease-duration (:lease_duration body)
     :data (:data body)}))

(defn read-secret [client path]
  (-> (wrap-request client path :get {})
      (wrap-response)))

(defn write-secret [client path value-map]
  (-> (wrap-request client path :put value-map)
      (wrap-response)))

(defn token-renew [client]
  (-> (wrap-request client "auth/token/renew" :put {:token (:token client)})
      (wrap-response)))

(defn token-lookup [client]
  (-> (wrap-request client "auth/token/lookup" :get {})
      (wrap-response)))

(defn maybe-renew-token [client]
  ;; TODO: make grace-period dynamically
  (let [grace-period 1000
        ttl (-> (token-lookup client) :data :ttl)]
    (if (< ttl grace-period)
      (do
        (println "renewing token....")
        (token-renew client)))))

(defn run-token-updater [client]
  (future
    (while true
      (Thread/sleep 5000)
      (maybe-renew-token client))))

;; TODO:
;; * caching
;; * refresh expired tokens?
;; * aws-ec2 auth backend support
