(ns circle-vault.core
  (:require [clojure.edn :as edn]
            [clojure.set :as set]
            [clojure.string :as string]

            [cheshire.core :as json]
            [clj-http.client :as http]
            [clj-time.core :as time]
            [com.stuartsierra.component :as component]))

(defn- role-id []
  (System/getenv "VAULT_ROLE_ID"))

(defn- security-id []
  (System/getenv "VAULT_SECURITY_ID"))

(defn- valid-path?
  [path]
  (and (string? path)
       (not (string/blank? path))))

(defmulti auth!
  (fn [url method data] method))

(defmethod auth! :default
  [url method data]
  (throw (ex-info (format "Authentication method %s not implemented" method) {})))

(defmethod auth! :app-id
  [url method {:keys [app-id user-id]}]
  {:pre [(string? app-id), (string? user-id)]}
  (let [response (http/post (str url "/v1/auth/app-id/login")
                            {:content-type :json
                             :body (json/generate-string {:app_id app-id, :user_id user-id})})
        body (json/parse-string (:body response) true)]
    (get-in body [:auth :client_token])))

(defmethod auth! :app-role
  [url method {:keys [role-id secret-id]}]
  {:pre [(string? role-id) (string? secret-id)]}
  (let [response (http/post (str url "/v1/auth/approle/login")
                            {:content-type :json
                             :body (json/generate-string {:role_id role-id
                                                          :secret_id secret-id})})
        body (json/parse-string (:body response) true)]
    (get-in body [:auth :client_token])))

(defn authenticate [vault-client]
  (auth! (:api-url vault-client)
         (:auth-method vault-client)
         (:auth-data vault-client)))


(defrecord VaultClient [api-url auth-method auth-data token]
  component/Lifecycle

  (start [this]
    (if-let [token (authenticate this)]
      (assoc this :token token)
      ;; TODO: what can we do with the component when this doesn't work?
      this))

  (stop [this]
    (assoc this :token nil)))

(defn new-vault-client
  [api-url auth-method auth-data]
  (map->VaultClient {:api-url api-url
                     :auth-method auth-method
                     :auth-data auth-data
                     :token nil}))

(defn vault-client
  [url role-id secret-id]
  (-> (new-vault-client url :app-role {:role-id role-id :secret-id secret-id}) (component/start)))

(defn read-secret [client path]
  {:pre [(valid-path? path)]}
  (let [response (http/get (str (:api-url client) "/v1/" path)
                           {:headers {"X-Vault-Token" (:token client)}})
        ;; This "mangles" the data in that every JSON key string is
        ;; keywordised
        body (json/parse-string (:body response) true)]
    {:lease-duration (:lease_duration body)
     :data (:data body)}))

(defn write-secret [client path value-map]
  {:pre [(valid-path? path)]}
  (let [response (http/put (str (:api-url client) "/v1/" path)
                           {:headers {"X-Vault-Token" (:token client)}
                            :content-type :json
                            :body (json/generate-string value-map)})
        body (json/parse-string (:body response) true)]))

(defn token-renew [client]
  (let [path "auth/token/renew"
        token (:token client)
        value-map {:token token}
        response (http/put (str (:api-url client) "/v1/" path)
                           {:headers {"X-Vault-Token" (:token client)}
                            :content-type :json
                            :body (json/generate-string value-map)})
        body (json/parse-string (:body response) true)]
    body))

(defn token-lookup [client]
  (let [path "auth/token/lookup"
        token (:token client)
        value-map {:token token}
        response (http/get (str (:api-url client) "/v1/" path)
                           {:headers {"X-Vault-Token" (:token client)}
                            :content-type :json
                            :body (json/generate-string value-map)})
        body (json/parse-string (:body response) true)]
    body))

(defn maybe-renew-token [client]
  ;; TODO: make grace-period dynamically
  (let [grace-period 1000
        ttl (-> (token-lookup client) :data :ttl)]
    (if (< ttl grace-period)
      (do
        (println "renewing token....")
        (token-renew client)))))

(defn run-token-updater [client]
  (future
    (while true
      (Thread/sleep 5000)
      (maybe-renew-token client))))

;; TODO:
;; * caching
;; * refresh expired tokens?
;; * aws-ec2 auth backend support
