(ns circle-util.timing
  (:require [clojure.tools.logging :refer (infof)]
            [clj-time.core :as time]
            [circle-util.time :refer (millis-between interval->duration)]))

(defmacro time-str
  "Logs 'str X ms"
  [str & body]
  `(let [start# (time/now)
         result# (do ~@body)
         end# (time/now)
         elapsed# (millis-between start# end#)]
     (infof "%s %sms" ~str elapsed#)
     result#))

(defmacro time-body
  "Logs how long it takes to execute an expr"
  [& body]
  `(time-str (str (quote ~body)) ~@body))

(defn elapsed-time
  "Takes a thunk, calls it and returns the elapsed time, as a joda duration"
  [f]
  (let [start (time/now)
        result (f)
        end (time/now)]
    (interval->duration (time/interval start end))))

(defn wrap-time
  "Wraps a fn to log how long it took to execute"
  [f]
  (fn [& args]
    (let [start (time/now)
          result (apply f args)
          end (time/now)
          elapsed (millis-between start end)]
      (infof "time %s %sms" f elapsed)
      result)))

(defmacro with-elapsed-time
  "Runs body as normal, returns elapsed time for how long it took to run"
  [& body]
  `(elapsed-time (fn [] ~@body)))

(defn time-send
  "Same as clojure.core/send, but wraps the fn to log how long it took to execute"
  [a f & args]
  (apply send a (wrap-time f) args))

(defn time-send-off [a f & args]
  (apply send-off a (wrap-time f) args))

(defmacro with-time-limited
  [timeout-ms timeout-value & body]
  `(let [f# (future ~@body)
         v# (deref f# ~timeout-ms ~timeout-value)]
     (when-not (future-done? f#)
       (future-cancel f#))
     v#))

(defn times->intervals-ms
  "If you record a sequence of timestamps with (time/now), this turns it into
   a sequence of ms durations from one time to the next."
  [times]
  (let [pairs (partition 2 1 times)]
    (map (fn [pair] (time/in-millis (time/interval (nth pair 0) (nth pair 1))))
         pairs)))
