(ns circle-util.time
  (:refer-clojure :exclude [min max])
  (:require [clojure.core.typed :as t]
            [clojure.string :refer (join)]
            [clj-time.core :as time]
            [clj-time.coerce :as time-coerce]
            [clj-time.format :as time-format]
            [inflections.core :refer (pluralize)])
  (:import (org.joda.time DateTime
                          Duration
                          DateTimeZone
                          ReadableInstant
                          format.DateTimeFormatter)))

(t/warn-on-unannotated-vars)

(defn date-time-in-tz
  "Construct a timezone in a specific timezone"
  ([dt tz-str]
     (time/to-time-zone dt (time/time-zone-for-id tz-str)))
  ([#^Integer year #^Integer month #^Integer day #^Integer hour #^Integer minute #^Integer second #^Integer millis tz]
     (DateTime. year month day hour minute second millis #^DateTimeZone tz)))

(defn time-with-offset
  "given a datetime, return a new time that replaces the hour and minute"
  [^DateTime dt hour minute]
  (-> dt
      (.withHourOfDay hour)
      (.withMinuteOfHour minute)))

(defn at-midnight
  "Return the dt at midnight, rounding backwards"
  [dt]
  (time/date-midnight (time/year dt)
                      (time/month dt)
                      (time/day dt)))

(defn period-to-s
  "Takes a joda period, returns a human readable string"
  [period]
  (.print (org.joda.time.format.PeriodFormat/getDefault) period))

(defn from-now
  "Takes a period, returns the interval starting from now"
  [period]
  (time/interval (time/now) (time/plus (time/now) period)))

(def period->interval from-now)

(defn period->secs [p]
  (-> p .toStandardDuration .getStandardSeconds))

(defn period->millis [p]
  (-> p .toStandardDuration .getMillis))

(defn duration->millis [d]
  (.getMillis d))

(defn to-millis
  "Takes a joda interval, returns a number of millis in the interval"
  [interval]
  (when-not (nil? interval)
    (.toDurationMillis interval)))

(defn period-to-millis
  "Converts a period to millis, assuming the period starts now"
  [period]
  (-> period from-now to-millis))

(defn to-epoch-millis [dt]
  (.getMillis dt))

(def ->epoch-millis to-epoch-millis)

(defn ->epoch-seconds [dt]
  (-> dt ->epoch-millis (/ 1000.0) (int)))

(defn epoch-seconds->date [seconds]
  (-> seconds (* 1000) time-coerce/from-long))

(t/ann ^:no-check custom-formatters (t/HMap :mandatory {:rfc1123 DateTimeFormatter
                                                        :github-pushed-at DateTimeFormatter}))
(def custom-formatters
  {:github-pushed-at (time-format/formatter "yyyy/MM/dd HH:mm:ss Z")
   :rfc1123 (time-format/formatter "EEE, dd MMM yyyy HH:mm:ss 'GMT'")})

(defn parse
  "Returns a DateTime instance (in the parsed timezone) time zone obtained by parsing the
   given string according to the given formatter. This function is a copy of clj-time.parse/parse, except that it uses WithOffsetParsed"
  ([#^DateTimeFormatter fmt #^String s]
     (.parseDateTime (.withOffsetParsed fmt) s))
  ([#^String s]
     (first
      (for [f (vals (merge time-format/formatters custom-formatters))
            :let [d (try (parse f s) (catch Exception _ nil))]
            :when d] d))))


(defn pretty-day
  "Given a DateTime, print 'today', 'yesterday', 'monday', 'last week' or 'may 25th'"
  [date]
  (condp > (-> date (time/interval (time/now)) time/in-days)
    1 "today"
    2 "yesterday"
    7 (-> date .dayOfWeek .getAsText)
    (str date)))

(defn pretty-hour
  "Given a DateTime, print 'X hours ago', 'yesterday', 'monday', 'last week' or 'may 25th'"
  [date]
  (let [hours (-> date (time/interval (time/now)) time/in-hours)]
    (condp > hours
      24 (format "%s hours ago" hours)
      (* 2 24) "yesterday"
      (* 7 24) (-> date .dayOfWeek .getAsText)
      (str date))))

(defn destruct
  "Returns a vector, the parts of the time, that can be applied to the date-time constructor"
  [dt]
  [(time/year dt)
   (time/month dt)
   (time/day dt)
   (time/hour dt)
   (time/minute dt)
   (time/second dt)
   (time/milli dt)])

(def time-length-map
  {:year 1
   :month 2
   :day 3
   :hour 4
   :minute 5
   :second 6
   :milli 7})

(defn round
  "Truncate the datetime down to the nearest period. period can be a keyword, one of: :year :month :date :hour :second"
  [dt period]
  (->> dt
      (destruct)
      (take (time-length-map period))
      (apply time/date-time)))

(defn duration
  "Returns the duration between two datetimes"
  ^Duration [dt-a dt-b]
  (.toDuration (time/interval dt-a dt-b)))

(defn millis->duration ^Duration [ms]
  (Duration. ms))

(defn interval->duration [i]
  (.toDuration i))

(defn duration>
  "True if this duration is longer than Period p"
  [d p]
  (> (duration->millis d)
     (-> p from-now to-millis)))

(defn duration< [d p]
  (not (duration> d p)))

(defn maybe-interval
  "Returns an interval, or nil when dt-b is before dt-a. Use when the two datetimes could be created by different machines and clock skew, or bugs, or races"
  [dt-a dt-b]
  (when (and dt-a dt-b (not (time/before? dt-b dt-a)))
    (time/interval dt-a dt-b)))

(defn millis-between
  "Given two instants, return the number of millis between them.
  If start or end are nil, 0 is returned."
  [^ReadableInstant start ^ReadableInstant end]
  (or (to-millis (maybe-interval start end))
      0))

(defn weekday? [time]
  (< (time/day-of-week time) 6))

(defn posix-now []
  (-> (time/now)
      time-coerce/to-long
      (/ 1000)
      int))

(t/ann timestamp-hour [DateTime -> DateTime])
(defn #^org.joda.time.DateTime hour-timestamp [timestamp]
  (.withTime timestamp
             (.getHourOfDay timestamp)
             0 0 0))

(defn duration->str
  "Format the given duration as a human-readable string"
  [^Duration d]
  (if (zero? (.getMillis d))
    "zero seconds"
    (letfn [(field [single n] (when (pos? n)
                                (pluralize n single)))]
      (join " "
        (take 2
              (filter identity [(field "day" (.getStandardDays d))
                                (field "hour" (mod (.getStandardHours d) 24))
                                (field "minute" (mod (.getStandardMinutes d) 60))
                                (field "second" (mod (.getStandardSeconds d) 60))
                                (field "milli" (mod (.getMillis d) 1000))]))))))
