(ns circle-util.test
  (:require [clojure.test :refer (is)]
            [circle-util.core :refer (apply-if)]
            [circle-util.except :refer (throwf)]
            [slingshot.slingshot :refer (try+)]))

(defn anything
  "Dummy expression. Used to assert that expr can do anything except
throw, i.e. (is (anything (foo)))"
  [expr]
  true)

(defmacro is=
  "(is (= expected actual))"
  [expected actual]
  `(is (= ~expected ~actual)))

(defmacro is-re
  "(is (re-find re expr))"
  [re expr]
  `(is (re-find ~re ~expr)))

(defmacro is-not
  "(is (not (expected actual))"
  [actual]
  `(is (not ~actual)))

(defmacro is-thrown+
  "Assert that an error matching a slingshot selector is thrown."
  [pattern expr]
  `(is (try+ ~expr
             false
             (catch ~pattern _# true))))

(defn id=
  "Assert that all the arguments are mongo objects, and they all have
   the same :_id."
  [& objects]
  ;; make sure none of these are nil.
  (and (every? (comp not nil?) objects)
       ;; Deref the ones that are refs.
       (let [deref'd (map #(apply-if (instance? clojure.lang.IRef %) deref %) objects)]
         ;; make sure they all have ':_id' set.
         (and (every? #(contains? % :_id) deref'd)
              ;; Make sure all the :_ids are equal.
              (apply = (map :_id deref'd))))))

(defn stateful-fn*
  "Takes a seq of no argument fns. Returns a new fn that calls each fn in turn."
  [fns]
  (let [state (atom fns)]
    (fn [& _]
      (if (not (zero? (count @state)))
        (let [f (first @state)]
          (try
            (apply f [])
            (finally
              (swap! state rest))))
        (throwf "no more values")))))

(defmacro thunk-exprs
  "Takes a seq of unevaluated exprs. Returns a seq of no argument fns, that call each of the exprs in turn"
  [exprs]
  (into [] (map (fn [v]
                  `(fn []
                     ~v)) exprs)))

(defmacro stateful-fn
  "Takes a seq of unevaluated expressions. Returns a function that evaluates and returns each expression in turn. Throws if stateful-fn is called more times than there are values provided."
  [& exprs]
  `(stateful-fn* (thunk-exprs ~exprs)))
