(ns circle-util.string
  (:refer-clojure :exclude (take contains?))
  (:require [clojure.core.typed :as t]
            [clojure.string :as string]
            [clojure.core.strint :refer (<<)]
            [circle-util.levenshtein :as levenshtein]
            hiccup.util))

(t/warn-on-unannotated-vars)

(def case-insensitive-alphabet (vec
                                (map char (concat (range (int \a) (inc (int \z)))
                                                  (range (int \0) (inc (int \9)))))))
(def case-sensitive-alphabet (vec
                              (concat case-insensitive-alphabet
                                      (map char (range (int \A) (inc (int \Z)))))))

(defn rand-char [& {:keys [alphabet]}]
  (let [alphabet (or alphabet case-insensitive-alphabet)]
    (get alphabet (rand-int (count alphabet)))))

(defn rand-str [len & {:keys [alphabet]}]
  (apply str (clojure.core/take len (repeatedly rand-char))))

(defn rand-str-lower [len]
  (rand-str len :alphabet case-insensitive-alphabet))

(defn first-name [name]
  (-> name (string/split #" ") first))

(t/ann drop-left [String String -> (t/Option String)])
(defn drop-left
  "if str starts with substr, return the part of str after substr

  (drop-left \"foobar\" \"foo\") => \"bar\""
  [^String str ^String substr]
  (when (.startsWith str substr)
    (.substring str (count substr))))

(defn drop-right
  "if str ends with substr, returns the part of str before substr

  (drop-right \"foobar\" \"bar\") => \"foo\""
  [^String str ^String substr]
  (when (.endsWith str substr)
    (.substring str 0 (- (count str) (count substr)))))

(defn escape+
  "Works the same as clojure.string/escape, but the key can be a string, rather than a char"
  [str map]
  (reduce (fn [str [key val]]
            (string/replace str key val)) str map))

(defn substring
  ([^String str start]
     (.substring str start))
  ([^String str start end]
     (.substring str start end)))

(defn contains?
  "True if s2 contains s1"
  [^String s1 ^String s2]
  (.contains s2 s1))

(defn closest
  "Find the closest string in haystack to the needle
  This can be used to suggest a command like git does:
    git: 'chekcout' is not a git command. See 'git --help'.
    Did you mean this?
      checkout
  Returns nil if haystack is empty"
  [needle haystack]
  (first (sort-by (partial levenshtein/distance needle) haystack)))

(defn indices-of
  "Return the indices of target in corpus"
  [^String corpus ^String target]
  (loop [i (.indexOf corpus target 0)
         indicies []]
    (if (= -1 i)
      indicies
      (recur (.indexOf corpus target (inc i)) (conj indicies i)))))

(t/ann take [String t/Int -> String])
(defn take
  "Returns a new string consisting of the first n characters"
  [^String str n]
  (subs str 0 (int (min n (count str)))))

(t/ann truncate [String t/Int -> String])
(defn truncate
  "Truncates a string to a certain length and adds \"...\" if necessary.
   Accounts for increased length of ellipses."
  [^String s n]
  (cond (< n 4) "" ;; doesn't really make sense to truncate a string to less than 4
        (< n (count s)) (str (take s (- n 3)) "...")
        :else s))

(t/ann ends-with? [String String -> Boolean])
(defn ends-with?
  "True if str ends with the specified suffix"
  [^String str ^String suffix]
  (.endsWith str suffix))

(t/ann starts-with? [String String -> Boolean])
(defn starts-with?
  "True if str begins with the specified prefix"
  [^String str ^String prefix]
  (.startsWith str prefix))

(t/ann all-caps [String -> String])
(defn all-caps [^String str]
  (.toUpperCase ^String str))

(t/ann snake_case->mixedCase [String -> String])
(defn snake_case->mixedCase [^String s]
  (let [words (.split s "_")]
    (string/join "" (into [(first words)]
                          (mapcat string/capitalize (rest words))))))

(t/ann spinal-case->SCREAMING_SNAKE_CASE [String -> String])
(defn spinal-case->SCREAMING_SNAKE_CASE [^String s]
  (when s
    (let [words (.split ^String (all-caps s) "-")]
      (string/join "_" words))))

(t/ann SCREAMING_SNAKE_CASE->spinal-case [String -> String])
(defn SCREAMING_SNAKE_CASE->spinal-case [^String s]
  (when s
    (let [words (.split ^String (.toLowerCase s) "_")]
      (string/join "-" words))))

(defn get-bytes
  "Returns the byte array for this string assuming UTF-8 encoding"
  [^String str]
  ;; UTF-8 only
  (.getBytes str "UTF-8"))

(defn append [s a]
  (str s a))

(defn prepend [s p]
  (str p s))

(defn url? [s]
  (try
    (java.net.URL. s)
    true
    (catch java.net.MalformedURLException e
      false)))

(defn remove-trailing
  "Remove all trailing occurences of c from s"
  ;;Examples:
  ;; (remove-trailing "foo" \o) => f
  ;; (remove-trailing "bar" \b) => bar
  [^String s ^Character c]
  (if (ends-with? s (str c))
    (recur (take s (-> s count dec)) c)
    s))

(defn repeat-char
  "Returns a string of the character c repeated n times"
  [n c]
  {:pre [(>= n 0)]}
  (if (> n 0)
    (apply str (repeat n c))
    ""))

(defmacro <<-escape-html [strint-str]
  `(hiccup.util/escape-html (<< ~strint-str)))

(defn format-escape-html [& args]
  (hiccup.util/escape-html (apply format args)))

(defn normalize-input
  "Returned a trimmed string or nil if blank string"
  [str]
  (let [normalized (some-> str string/trim)]
    (when (seq normalized)
      normalized)))

(defn string->keyword
  "DWIM when I ask for the keyword version of a string.

   nil -> nil
   :foo -> :foo
   \"foo\" -> :foo
   \":foo\" -> :foo"
  [string]
  {:pre [(or (nil? string) (string? string) (keyword? string))]}
  (cond (nil? string) nil
        (keyword? string) string
        :else (if (= \: (first string))
                (keyword (subs string 1))
                (keyword string))))
