(ns circle-util.ssh-keys
  (:require [clojure.core.typed :as t]
            [clojure.string :as str]
            [clj-keyczar.crypt :as crypt]
            [clj-ssh.ssh :as ssh]
            [circle-util.sh :as sh]
            [circle-util.fs :refer (with-delete with-temp-file)]
            [circle-util.core :refer (apply-if)]
            [circle-util.crypto :as crypto]))

(t/warn-on-unannotated-vars)

(t/defalias Key String)
(t/defalias KeyPair (t/HMap :mandatory {:public-key Key :private-key Key}
                            :optional {:hostname String}
                            :complete? true))
(t/defalias EncryptedKeyPair (t/HMap :mandatory {:public-key Key :encrypted-private-key crypto/CipherText}
                                     :optional {:hostname String}
                                     :complete? true))

(t/ann encrypt-keypair [KeyPair -> EncryptedKeyPair])
(defn encrypt-keypair [keypair]
  (let [private-key (:private-key keypair)]
    (assoc (dissoc keypair :private-key)
           :encrypted-private-key (crypt/encrypt private-key))))

(t/ann decrypt-keypair [EncryptedKeyPair -> KeyPair])
(defn decrypt-keypair [encrypted-keypair]
  (let [encrypted-private-key (:encrypted-private-key encrypted-keypair)]
    (assoc (dissoc encrypted-keypair :encrypted-private-key)
           :private-key (crypt/decrypt encrypted-private-key))))

(defn- add-hostname
  "Adds the hostname comment to the end of a public key"
  [pub-key hostname]
  (str/replace pub-key #"\n$" (format "%s\n" hostname)))

(t/ann ^:no-check generate-keys [& :optional {:hostname String} -> KeyPair])
(defn generate-keys
  "Generate a new pair of SSH keys, returns a map"
  [& {:keys [hostname]}]
  (let [ssh-agent (ssh/ssh-agent {:use-system-ssh-agent false
                                  :known-hosts-path nil})
        [priv pub] (ssh/generate-keypair ssh-agent :rsa 2048 nil)
        pub (String. pub)
        pub (apply-if hostname add-hostname pub hostname)]
    {:private-key (String. priv)
     :public-key pub}))

(defn fingerprint
  "Returns the SSH fingerprint for a public key"
  [^String key]
  (when key
    (when-let [keypair (com.jcraft.jsch.KeyPair/load nil nil (.getBytes key "ASCII"))]
      (.getFingerPrint keypair))))

(defn- strip-hostname
  "Given an ssh public key string, strip the hostname from it"
  [pub-key]
  (-> pub-key
      (str/split #" ")
      (->>
       (take 2)
       (str/join " "))))

(defn compute-public-key
  "Compute public key associated with private key.
  The public key is returned as a string.
  This function does not handle password-protected private keys.
  Throws an unspecified Exception if the private key is password
  protected or if there is an IO error when computing the key."
  [private-key-str]
  (when (and (string? private-key-str)
             (seq private-key-str))
    (let [private-key-file (fs/tempfile)]
      (sh/shq! (chmod 600 ~private-key-file))
      (with-delete private-key-file
        (spit private-key-file (str/trim private-key-str))
        ;; Unless DISPLAY="" is set when calling ssh-keygen, the password input
        ;; will be read from stdin, which will block the process when running an
        ;; app from the terminal.
        ;; By setting  DISPLAY="" the SSH_ASKPASS program is called.
        ;; The `echo` program is used as a proxy to generate the empty string as
        ;; input, since SSH_ASKPASS needs to be an executeable program.
        (let [{:keys [out]} (sh/shq! ("SSH_ASKPASS=echo" "DISPLAY=\"\"" ssh-keygen -y -f ~private-key-file))]
          (str/trim out))))))

(defn keys-match?
  "True if these public and private keys are belong to each other"
  [public-key-str private-key-str]
  (when (and
          (string? public-key-str)
             (string? private-key-str)
             (seq public-key-str)
             (seq private-key-str))
    (with-temp-file private-key-file
      (spit private-key-file (str/trim private-key-str))
      (let [{:keys [exit out err]} (sh/sh!  (sh/q ("SSH_ASKPASS=/dev/null" ssh-keygen -y -f ~private-key-file)))]
        (= (strip-hostname (str/trim out)) (strip-hostname (str/trim public-key-str)))))))
