(ns circle-util.sh
  "fns for running code locally, and compiling stevedore to bash"
  (:require [clojure.core.typed :as t :refer (AnyInteger Option)]
            [clojure.java.io :as io]
            [clojure.java.shell]
            [clojure.string :as str]
            [clojure.tools.logging :as logging]
            [pallet.stevedore :as stevedore]
            [pallet.stevedore.bash :as bash]
            [fs :as fs]
            [circle-util.core :refer (apply-map to-name)]
            [circle-util.except :refer (throw-if-not+ throw-if-not eat)]
            [circle-util.time :as c-time])
  (:import java.io.InputStream
           clojure.lang.Namespace
           clojure.tools.logging.impl.Logger
           org.joda.time.PeriodType))

(defmacro q
  "Quasiquote a seq of stevedore forms, ((rvm use foo) (bash bar))"
  [& forms]
  (assert (seq forms) "missing commands")
  (assert (every? seq? forms) "stevedore forms are expected to be, well, forms...")
  `(let [resp# (stevedore/quasiquote (~@forms))]
     (assert (seq resp#))
     resp#))

;; Documenting the issues that the following Steve* types have but not fixing
;; them, the intent of the code as written calls for enforcement of constraints
;; by core.typed that are not currently possible.
;;
;; These types are inaccurate and unusable because of how the `q` macro
;; operates.
;;
;; `q` wraps its argument in parens, e.g (q (rvm use foo)) results in ((rvm use
;; foo)), so the definition of a SteveOneLiner is for a value that can't be
;; generated.
;; The functions that are typed as accepting one-liners do *not* actually
;; accept a SteveOneLiner.
;;
;; A one-liner is actually a list of a single list of symbols/strings/integers
;; E.g. ((rvm use foo))
;;
;; A multi-liner is a list of lists of symbols/strings/integers
;; E.g. ((rvm use foo) (rvm use bar))
;;
;; Notably, by these definitions, a multi-liner is *not* a sequence of
;; one-liners (contradicting the SteveMultiLine type alias).
;; You can compensate by introducing SteveBody which represents the "inner"
;; list of a one-liner or a component of a multi-liner.
;;
;; A one-liner is then a sequence of exactly one SteveBody and a multi-liner is
;; a sequence of at least two SteveBodys
;;
;; Strings slightly complicate matters, the following are all valid:
;; 1) "rvm use foo"                  - a one-liner
;; 2) ["rvm use foo"]                - also a one-liner
;; 3) ["rvm use foo", "rvm use bar"] - a multi-liner
;;
;; Cases 2 and 3 are handled by unioning String into the defintion of SteveBody
;; Case 1 is handled by unioning String into the defintion of SteveOneLiner
;;
;; This results in a more accurate set of types:
;; (t/defalias SteveBody (t/U String (t/Seqable (t/U String t/Sym t/AnyInteger))))
;; (t/defalias SteveOneLiner (t/U String (t/HSequential [SteveBody])))
;; (t/defalias SteveMultiLine (t/I (t/Seqable SteveBody) (t/CountRange 2))
;; (t/defalias SteveExpr (t/U String SteveOneLiner SteveMultiLine))
;;
;; These still don't work because core.typed can't statically infer that
;; `(sh/q (rvm use foo))`
;; will result in a different type to
;; `(sh/q (rvm use foo) (echo bar))`
;;
;; In both cases it infers a list of a list of symbols/strings.
;;
;; As a result it's impossible to ask core.typed to enforce anything to do with
;; SteveOneLiner types.
(t/defalias SteveOneLiner (t/U String (t/Seqable (t/U String t/Sym t/AnyInteger))))
(t/defalias SteveMultiLine (t/Seqable SteveOneLiner))
(t/defalias NonEmptySteveMultiLine (t/NonEmptySeqable SteveOneLiner))
(t/defalias SteveExpr (t/U String SteveMultiLine))

(t/defalias ShMap (t/HMap :mandatory
                          {:exit t/Int
                           :out t/Str
                           :err t/Str}))

(t/ann q-arg [String -> String])
(defn q-arg
  "Wraps a (potentially untrusted) string up as a single command-line argument, e.g.:
    foo;rm -rf *  ->  'foo;rm -rf *'"
  [in]
  (format "'%s'" (str/replace in "'" "'\\''")))

(t/ann qq-arg [String -> String])
(defn qq-arg
  "Wraps a (potentially untrusted) string up as a single command-line argument, while
  allowing shell interpolation of the contents, e.g.:
    $PATH:/usr/bin/ -> \"$PATH:/usr/bin\""
  [in]
  (format "\"%s\"" (str/replace in "\"" "\\\"")))

(t/defalias EnvMap (t/Map (t/U String t/Kw) String))
(t/ann ^:no-check format-bash-cmd [(t/U SteveExpr SteveOneLiner) (Option EnvMap) (Option String) -> SteveExpr])
(defn format-bash-cmd [body environment pwd]
  (let [cd-form (when (seq pwd)
                  (q (cd ~(qq-arg pwd))))
        env-form (map (fn [[k v]]
                        (format "export %s=%s" (to-name k) (qq-arg (to-name v)))) environment)]
    (concat cd-form env-form body)))

(t/ann pallet.stevedore/chain-commands [SteveExpr * -> String])
(defmacro q-chain
  "Same as q, but use '&&' between commands, rather than ';'"
  [& forms]
  `(stevedore/with-script-language ::bash/bash
     (stevedore/chained-script ~@forms)))

(defmacro q-or
  "Same a q, but use '||' between commands.
  Doesn't handle commands separated by newlines."
  [& forms]
  `(q (~@(interpose "||" (list* forms)))))

;; XXX Definition of SteveOneLiner is not compatible with this fn
(t/ann ^:no-check q-concat* [SteveOneLiner SteveOneLiner -> SteveExpr])
(defn q-concat*
  [a b]
  (throw-if-not (= 1 (count a)) "q-concat only supports one liners")
  (throw-if-not (= 1 (count b)) "q-concat only supports one liners")

  (list (list* (concat (first a) (first b)))))

;; XXX Definition of SteveOneLiner is not compatible with this fn
(t/ann ^:no-check q-concat [SteveOneLiner * -> SteveExpr])
(defn q-concat
  "Takes N one-line stevedore forms. Returns a new one-line form, of the args
  concatenated.
  (q-concat (sh/q (foo bar)) (sh/q (baz cat))) => (sh/q (foo bar baz cat))"
  [arg & args]
  (reduce q-concat* arg args))

(t/ann pallet.stevedore/*script-language* (Option t/Kw))
(t/ann pallet.stevedore/*script-file* (Option String))
(t/ann pallet.stevedore/*script-line* (Option AnyInteger))
(t/ann pallet.stevedore/*script-ns* (Option Namespace))
(t/ann pallet.stevedore/emit-script [SteveExpr -> String])

(t/ann emit-form [(t/U SteveExpr SteveOneLiner)
                  & :optional {:environment (Option EnvMap)
                               :pwd (Option String)}
                  -> String])
(defn emit-form
  "Takes quoted stevedore code or a bash literal. Emits bash code, adding pwd
  and environment variables if necessary"
  [body & {:keys [environment
                  pwd]}]
  (let [body (if (string? body)
               [body]
               body)
        body (format-bash-cmd body environment pwd)]
    (stevedore/with-script-language ::bash/bash
      (binding [stevedore/*script-ns* *ns*]
        (stevedore/emit-script body)))))

(t/ann ^:no-check make-bash-script [String -> Process])
(defn- make-bash-script
  "creates a process to run a bash script file."
  [script]
  (.exec ^Runtime (Runtime/getRuntime)
         ^"[Ljava.lang.String;" (into-array String ["bash" script])))


(t/ann process [String
                & :optional {:in java.io.InputStream
                             :in-enc String
                             :environment EnvMap
                             :pwd String
                             :log t/Kw}
                -> Process])
(defn ^Process process
  "Like sh, but returns the process object, which can be used to safely stop the
  thread. Script is a bash script file path. environment is a map of strings to
  strings."
  [script & {:keys [in in-enc]}]
  (let [^Process proc (make-bash-script script)]
    (if in
      (future
        (with-open [os (.getOutputStream proc)]
          (io/copy ^String in os :encoding (or in-enc "UTF-8"))))
      (-> proc .getOutputStream .close))
    proc))

(t/ann process-exit-code [Process -> (Option AnyInteger)])
(defn process-exit-code
  "Returns the process's exit code, or nil if it isn't finished yet"
  [^Process p]
  (try
    (.exitValue p)
    (catch java.lang.IllegalThreadStateException e
      nil)))

(t/ann process-kill [Process -> nil])
(defn process-kill [^Process p]
  (.destroy p))

(t/ann ^:no-check sh [SteveExpr
                      & :optional {:environment (Option EnvMap)
                                   :pwd (Option String)
                                   :log Logger
                                   :timeout PeriodType}
                      -> ShMap])
(defn sh
  "Runs code in a subprocess. Body can be a string or stevedore code.
  Returns a map."
  [body & {:keys [environment pwd log timeout in]
           :as args}]
  ;; Create a bash script containing commands to ensure that all commands can
  ;; be handled properly.
  (let [script (fs/tempfile)]
    (try
      (let [full-cmd (emit-form body :environment environment :pwd pwd)
            _ (spit script full-cmd)
            timeout-ms (some-> timeout c-time/period->millis)
            proc ^Process (apply-map process script args)
            exit-promise (promise)]

        (when log
          (logging/log log (format "%s => %s" full-cmd proc)))

        (with-open [out-stream (.getInputStream proc)
                    err-stream (.getErrorStream proc)
                    out (java.io.StringWriter.)
                    err (java.io.StringWriter.)]

          (future (deliver exit-promise (.waitFor proc)))

          (let [out-f (future (io/copy out-stream out :encoding "UTF-8") (.toString out))
                err-f (future (io/copy err-stream err :encoding "UTF-8") (.toString err))
                exit (if-not timeout
                       (deref exit-promise)
                       (deref exit-promise timeout-ms :timed-out))

                resp (if (not= :timed-out exit)
                       {:exit exit :out @out-f :err @err-f}
                       (do
                         ;; kill and return whatever is read so far
                         (process-kill proc)
                         (.waitFor proc) ;; wait for signal exit code, typically 143
                         {:exit (process-exit-code proc) :out (.toString out) :err (.toString err)}))]
            (when log
              (logging/log log (format "%s => %s" proc resp)))
            resp)))
      (finally
        (fs/delete script)))))

(t/ann ^:no-check sh! [SteveExpr & :optional {:in InputStream :log t/Kw} -> ShMap])
(defn sh!
  "takes the output of sh. throws if (not= :exit 0). :log takes a
  keyword, a log-level recognized by clojure.tools.logging for logging
  the in & output of the command"
  [body & {:as opts}]
  (let [resp (apply-map sh body opts)
        zero-resp? (-> resp :exit zero?)]
    (throw-if-not+ zero-resp? resp "%s returned %s: %s %S" body (-> resp :exit) (-> resp :out) (-> resp :err))
    resp))

(defmacro shq
  "like sh, but quasiquotes its arguments"
  [body & {:keys [environment pwd]}]
  `(sh (q ~body) :environment ~environment :pwd ~pwd))

(defmacro shq!
  [& forms]
  `(sh! (q ~@forms)))


(t/defalias BackgroundProcInfo (t/HMap :mandatory
                                       {:proc Process
                                        :tmpfiles (t/Seqable String)}))

(t/ann ^:no-check start-background-proc [SteveExpr -> BackgroundProcInfo])
(defn start-background-proc [body]
  ;; Hacky function mainly to be used for tests
  (let [script (fs/tempfile)]
    (try
      (let [full-cmd (emit-form body)
            _ (spit script full-cmd)
            proc (process script)]
        {:proc proc :tmpfiles [script]}))))

(t/ann ^:no-check destroy-background-proc [SteveExpr -> t/Any ])
(defn destroy-background-proc [proc-info]
  (eat (.destroy (:proc proc-info)))
  (eat (.waitFor (:proc proc-info)))
  (eat
   (doseq [f (:tmpfiles proc-info)]
     (fs/delete f))))
