(ns circle-util.seq
  "fns that work on seqs"
  (:require [clojure.core.typed :as t]))

(t/warn-on-unannotated-vars)

(t/ann find-first
       (t/All [x y]
         (t/IFn
           [(t/IFn [x -> t/Any :filters {:then (is y 0), :else tt}]) (t/Nilable (t/Seqable x)) -> (t/Option y)]
           [(t/IFn [x -> t/Any :filters {:then (! y 0), :else tt}]) (t/Nilable (t/Seqable x)) -> (t/Option (t/I (Not y) x))]
           [(t/IFn [x -> t/Any]) (t/Nilable (t/Seqable x)) -> (t/Option x)])))

(defn find-first
  "Returns the first item that passes predicate f"
  [f seq]
  (->> seq
       (filter f)
       (first)))

(defn concatv
  "Same as concat, but returns a vector"
  [& args]
  (->> args
       (apply concat)
       (into [])))

(defn indexed
  "returns a seq of [index item] pairs"
  [coll]
  (map vector (range) coll))

(defn index-of
  "Given a coll, returns the index of the item that is = to val. O(n)"
  [coll val]
  (->> coll
       (indexed)
       (filter #(= val (second %)))
       (first)
       (first)))

(defn filter-indexed
  "Returns all items where (f index item) return true"
  [f coll]
  (keep-indexed (fn [index item]
                  (if (f index item)
                    item
                    nil)) coll))

(defn remove-indexed
  [f coll]
  (filter-indexed (complement f) coll))

(defn remove=
  "remove vals = v"
  [v coll]
  (remove #(= v %) coll))

(defn into-map [args]
  (apply hash-map args))

(t/ann ^:no-check partition-into (t/All [x] [t/Int (t/Nilable (t/Seqable x)) -> (t/Seq (t/Vec x))]))
(defn partition-into
  "Splits the coll into n equal-sized seqs."
  [n coll]
  (let [counter (t/ann-form (atom -1) (t/Atom1 t/Int))]
    (->> (group-by (t/fn [_ :- x]
                     (swap! counter inc)
                     (mod @counter n)) coll)
         (merge (apply merge (t/for [i :- t/Int (range n)]
                               :- (t/Map t/Int (t/Vec x))
                               {i []}))) ; always return n seqs!
         (vals))))

(defn insert-before
  "Inserts a single item into the seq, once, the first time (f item) returns true."
  [pred item coll]
  ((fn inner [coll]
    (lazy-seq
     (when-let [[x & xs] (seq coll)]
       (if (pred x)
         (cons item (cons x xs))
         (cons x (inner xs))))))
   coll))

(defn insert-after
  "Inserts a single item into the seq, once, after the first time (f item) returns true."
  [pred item coll]
  ((fn inner [coll]
    (lazy-seq
     (when-let [[x & xs] (seq coll)]
       (if (pred x)
         (cons x (cons item xs))
         (cons x (inner xs))))))
   coll))

(defn insert-seq-after
  "same as insert-after, but inserts a seq of items rather than a single item"
  [pred s coll]
  ((fn inner [coll]
    (lazy-seq
     (when-let [[x & xs] (seq coll)]
       (if (pred x)
         (cons x (concat s xs))
         (cons x (inner xs))))))
   coll))

(defn drop-duplicates
  "Given two seqs, drops the head of both (while (= (first a) (first b))). Returns a vector of two seqs"
  [a b]
  (loop [a a
         b b]
    (if (and (seq a)
             (seq b)
             (= (first a)
                (first b)))
      (recur (rest a)
             (rest b))
      [a b])))

(defn any?
  "True if any item in pred is truthy for any item in seq. Same as 'some', but returns a boolean, and the name is better "
  [pred coll]
  (boolean (some pred coll)))

(defn any?=
  "True if any item in coll = val"
  [val coll]
  (some #(= val %) coll))

(defn any-re-find?
  "True if any item in coll matches the regex"
  [re coll]
  (some #(re-find re %) coll))

(defn separate
  "Returns a vector of two seqs, the seq of elements where (pred x) is true, and the seq where (pred x) is false"
  [pred coll]
  ((juxt filter remove) pred coll))

(defn removev [pred coll]
  (filterv (complement pred) coll))

(defn zip [& seqs]
  (apply map vector seqs))

(defn histogram
  "Returns a map containing a key for each value in coll, and the number of times that value appeared in coll"
  [coll]
  (reduce (fn [final item]
            (update-in final [item] (fnil inc 0))) {} coll))

(defn heads-match?
  "True if a equals the head of b"
  [a b]
  (= a (take (count a) b)))

(defn sublist?
  "True if seq a appears anywhere in b"
  [a b]
  (loop [a a
         b b]
    (if (heads-match? a b)
      true
      (when (seq b)
        (recur a (rest b))))))

(defn interleave-all
  "Like interleave, but consumes all seqs"
  [& colls]
  (let [colls (remove empty? colls)]
    (if (empty? colls)
      ()
      (concat (map first colls)
              (lazy-seq (apply interleave-all (map rest colls)))))))

(defn min-index
  "Takes a seq of numbers. Returns the index of coll which is lowest"
  [coll]
  (->> coll
       (map-indexed vector)
       (apply min-key second)
       (first)))


(defn distribute-key
  "Split a seq of ints into n approximately-even seqs. Adds
  the next item into the coll with the lowest score"
  [score-fn n coll]
  (let [coll (->> coll (sort-by score-fn) reverse)
        scores (atom (vec (repeat n 0)))]
    (reduce (fn [ret item]
              (let [i (min-index @scores)]
                (swap! scores update-in [i] + (score-fn item))
                (update-in ret [i] conj item))) (vec (repeat n [])) coll)))

(defn split-groups
  "Split a seq with the given predicate. The first value is the vector of positive results."
  [pred coll]
  {:post [(= (count (apply concat %)) (count coll))]}
  (let [{truthy true, falsy false} (group-by (comp boolean pred) coll)]
    [truthy falsy]))

(defn dequeue!
  "Transactionally dequeues queue head from queue atom"
  [queue]
  (loop []
    (let [q     @queue
          value (peek q)
          nq    (pop q)]
      (if (compare-and-set! queue q nq)
        value
        (recur)))))

(defn move-to-first
  "returns a seq with the first item in coll for which pred is true at the front

  returns coll unchanged if no items match pred"
  [pred coll]
  (let [[items-before [matching-item & items-after :as items-not-before]]
        (split-with (complement pred) coll)]
    (if (empty? items-not-before)
      coll
      (concat [matching-item] items-before items-after))))
