(ns circle-util.semantic-version
  (:refer-clojure :exclude [compare == < > <= >=])
  (:require [clojure.string :as str]
            [clojure.core.typed :as t]
            [clojure.core.logic :refer :all]
            [coercer.core :refer (coerce)]))

(t/ann ^:no-check parse [String -> (t/Seqable (t/U Integer String))])
(defn parse
  "Parses a semantic version string, like '1.2.3'. Returns a seq of integers"
  [v]
  (-> v
      (str/split #"\.")
      (->>
       (map #(or (coerce % Integer) %)))))

(t/ann ^:no-check compare [String String -> Integer])
(defn compare
  "a & b are two strings that are semantic versioned. Returns a negative integer, 0, or a positive one, for a < b, a = b, a > b"
  [a b]
  (loop [a (parse a)
         b (parse b)]
    (if (and (nil? a) (nil? b))
      0
      (let [[a-seg & rst-a] a
            [b-seg & rst-b] b
            result (clojure.core/compare a-seg b-seg)]
        (if (zero? result)
          (recur rst-a rst-b)
          result)))))

(t/ann <= [String String -> Boolean])
(defn <= [a b]
  (clojure.core/<= (compare a b) 0))

(t/ann >= [String String -> Boolean])
(defn >= [a b]
  (clojure.core/>= (compare a b) 0))

(t/ann < [String String -> Boolean])
(defn < [a b]
  (neg? (compare a b)))

(t/ann > [String String -> Boolean])
(defn > [a b]
  (pos? (compare a b)))

;; TODO: Add proper types here
(t/ann ^:no-check matches? [(t/IFn [t/Any t/Any t/Any -> t/Any]) String -> Boolean])
(defn matches?
  "True if version v, a semver string, matches the constraints

  constraints is a fn that takes three args, major, minor, patch, and returns a core.logic expression"
  [constraints v]
  (boolean
   (seq
    (run* [q]
      (fresh [v-major v-minor v-patch
              q-major q-minor q-patch]
             (conso v-major [v-minor v-patch] (parse v))
             (== v-major q-major)
             (== v-minor q-minor)
             (== v-patch q-patch)
             (constraints q-major q-minor q-patch))))))
