(ns circle-util.sed
  "fns that generate sed strings. Useful for passing to pallet or bash"
  (:require [clojure.string :as str]
            [clojure.core.typed :as t]
            [circle-util.sh :refer (q-arg)]))

(t/warn-on-unannotated-vars)

(t/ann sed-escape-map (t/Map Character String))
(def sed-escape-map {\newline "\\n"
                     \' "'\"'\"'" ;; http://stackoverflow.com/questions/1250079/bash-escaping-single-quotes-inside-of-single-quoted-strings
                     \/ "\\/"
                     })

(defn insert-line
  "Returns the sed command for insert 'line' in 'file' before the line that matches 'before'"
  [file line before]
  (format "sed '/%s/ i\\
%s' -i %s" (str/escape before sed-escape-map) line file))

(t/ann replace-line [String String String -> String])
(defn replace-line
  "Returns the sed for replace the line that matches 'match' with 'replacement' in 'file'"
  [file match replacement]
  (format "sed -i '/%s/ c\\
%s
' %s" (str/escape match sed-escape-map) (str/escape replacement sed-escape-map) file))

(defn substitute-all
  [file match replacement & {:keys [separator]}]
  (let [separator (or separator "/")]
    (format "sed -i 's%s%s%s%s%sg' %s" separator match separator replacement separator file)))

(defn append-file
  [file text]
  (format "echo %s >> %s" (q-arg (str text "\n")) file))

(defn overwrite-file
  "Sets to content of file to text.

  This method is meant for text only and appends a trailing newline to the file content."
  [file text & {:keys [eof-marker]
                :or {eof-marker "EOF"}}]
  (format "( cat <<'%s'\n%s\n%s\n ) > %s" eof-marker text eof-marker file))
