(ns circle-util.secret-keeper.versioned
  (:require [circle-util.secret-keeper :as secret-keeper]))

;; Library component
(defn add-migration
  "Helper function for adding a migration - automatically update version as well"
  [migrations version func]
  (conj migrations [version func]))

(defn applicable-migrations [curr-version all-migrations]
  (->> all-migrations
       (sort-by first)
       (filter (fn [[v m]] (> v curr-version)))))

(defn run-migrations
  [secrets migrations]
  (let [version (or (:version secrets) 0)
        applicable-migrations (applicable-migrations version migrations)
        apply-fn (fn [secrets version-migration]
                   (let [[version migration-fn] version-migration]
                     (-> (migration-fn secrets)
                         (assoc :version version))))]
    (reduce apply-fn secrets applicable-migrations)))

(defrecord VersionedSecretsStorage [underlying-storage migrations]
  ;; A versioned secrets storage that apply upgrade migrations before secrets
  ;; get validated.
  ;; The underlying secrets are expected to have a `version` field with a
  ;; strict total ordering."
  secret-keeper/SecretsStorage
  (read [this]
    (-> (secret-keeper/read underlying-storage)
        (run-migrations migrations)))
  (write [this secrets]
    (secret-keeper/write underlying-storage secrets)))
