(ns circle-util.secret-keeper
  (:refer-clojure :exclude [read])
  (:require [clojure.edn :as edn]
            [clojure.pprint :refer (pprint)]
            [schema.core :as s]
            [fs :as fs]
            [circle-util.core :refer (apply-map to-name)]
            [circle-util.sh :as sh]
            [circle-util.fs :refer (with-temp-file)]
            [circle-util.gpg-agent :as gpg-agent]
            [clojure.java.io :as io]
            [slingshot.slingshot :refer (throw+)])
  (:import clojure.lang.ExceptionInfo java.net.URL))

(defprotocol SecretsStorage
  (read [this] "Returns a secrets map from storage")
  (write [this secrets] "Writes a secrets map to storage"))

(defrecord LocalFileSecretsStorage [path]
  SecretsStorage
  (read [this]
    (edn/read-string (slurp (:path this))))
  (write [this secrets]
    (with-temp-file tmp-path
      (spit tmp-path (with-out-str (pprint secrets)))
      (fs/rename tmp-path path))))

(defrecord EncryptedLocalFileSecretsStorage [path passphrase]
  SecretsStorage
  (read [this]
    (let [resp (sh/sh (sh/q (gpg --batch --no-tty --decrypt --passphrase-fd 0 ~path))
                      :in (format "%s\n" passphrase))]
      (when (not (zero? (:exit resp)))
        (throw (ex-info (:err resp) resp)))
      (-> resp
          :out
          (edn/read-string))))
  (write [this secrets]
    (with-temp-file tmp-clear
      (let [tmp-crypt (str tmp-clear ".gpg")]
        (try
          (spit tmp-clear (with-out-str (pprint secrets)))
          (sh/sh! (sh/q (gpg --batch --no-tty --symmetric --passphrase-fd 0 --passphrase-repeat 0 --output ~tmp-crypt ~tmp-clear))
                  :in (format "%s\n" passphrase))
          (sh/sh! (sh/q (mv ~tmp-crypt ~path)))
          (finally
            (fs/delete tmp-crypt)))))))

(defmethod clojure.core/print-method EncryptedLocalFileSecretsStorage [x writer]
  (.write writer (str x)))

(defrecord EncryptedResourceStorage [resource passphrase]
  ;; load secrets from a .jar resource.
  SecretsStorage
  (read [this]
    (with-temp-file tmp-path
      (with-open [resource-stream (.openStream resource)]
        (io/copy resource-stream (java.io.File. tmp-path))
        (read (->EncryptedLocalFileSecretsStorage tmp-path passphrase)))))
  (write [this secrets]
    ;; n.b. this only works if the resource resolves to a file:/// URI... and that's good!
    (write (->EncryptedLocalFileSecretsStorage (.getFile resource) passphrase) secrets)))

(defmethod clojure.core/print-method EncryptedResourceStorage [x writer]
  (.write writer (str x)))

(defrecord PlainTextResourceStorage [resource]
  ;; load secrets from a .jar resource.
  SecretsStorage
  (read [this]
    (-> (.openStream resource)
        slurp
        (edn/read-string)))
  (write [this secrets]
    (spit resource (with-out-str (pprint secrets)))))

(defmethod clojure.core/print-method PlainTextResourceStorage [x writer]
  (.write writer (str x)))

(defrecord CodeSecretsStorage [value]
  SecretsStorage
  (read [this]
    (:value this))
  (write [this secrets]
    (throw (UnsupportedOperationException. "nope"))))

(defn validate-secrets
  "Takes some secrets and a schema. Runs schema/validate, but redacts the actual contents
  of the secrets from the thrown exception, if validation fails."
  [secrets-schema secrets]
  (try
    (s/validate secrets-schema secrets)
    (catch ExceptionInfo e
      (throw (ex-info (.getMessage e)
                      (assoc (.getData e) :value "redacted"))))))

(defn load-secrets
  "Takes a SecretsStorage and returns a secrets map"
  [store secrets-schema]
  (let [secrets (read store)]
    (validate-secrets secrets-schema secrets)))

(defn set-secrets!
  "Takes a SecretsStorage and rewrites the stored secrets to those provided."
  [store secrets-schema secrets]
  (validate-secrets secrets-schema secrets)
  (write store secrets)
  secrets)

(defn alter-secrets!
  "Takes a SecretsStorage and rewrites the secrets to the value:

  (apply f old-secrets args)"
  [store secrets-schema f & args]
  (set-secrets! store secrets-schema (apply f (read store) args)))

(defn copy-secrets!
  "Takes two SecretsStorage instances, and writes the secrets from one to the other. Useful,
  e.g., to change the passphrase of encrypted secrets storage."
  [orig-store dest-store]
  (->> orig-store
       read
       (write dest-store)))

(defn check-readable
  "Make sure some given secrets are readable; request a new passphrase if not. "
  [storage]
  (try (read storage)
       storage
       (catch Throwable ex
         (throw+ {:incorrect-passphrase true
                  :exception ex}))))

(defn uri-secrets
  "Get secrets from a URI defined by CIRCLE_SECRETS_URI."
  [& {:keys [passphrase]}]
  (when-let [uri (System/getenv "CIRCLE_SECRETS_URI")]
    (let [file (or (io/resource uri) (URL. uri))
          description (format "Secrets for %s." uri)]
      (if passphrase
        (check-readable (->EncryptedResourceStorage file passphrase))
        (gpg-agent/with-passphrase "CIRCLE_SECRETS_URI" description passphrase
          (check-readable (->EncryptedResourceStorage file passphrase)))))))

(defn valid-paths
  "Choose some paths to check for secrets, ordered by preference."
  [& {:keys [env job-name path prefix]
      :or {prefix ["secrets"]}}]
  (if path
    ;; if the path is specified, use only the path.
    [path]
    (->> ;; prefer to use env and job-name
          [(when (and env job-name)
             (format "%s-%s-secrets.edn.gpg"
                     (to-name env)
                     (to-name job-name)))
           ;; but we may also use just env
           (when env
             (format "%s-secrets.edn.gpg"
                     (to-name env)))]
          ;; delete nils
          (remove nil?)
          ;; and apply the prefix
          (map #(reduce fs/join (conj prefix %))))))

(defn file-secrets
  "Get secrets from a file, either in 'secrets' with a name derived
   from the given env and job-name, or with a given path."
  [& {:keys [env job-name path passphrase] :as args}]
  (let [path (->>  (apply-map valid-paths args) (filter fs/exists?) (first))
        message (format "Passphrase for %s." path)]
    (when path
      (if passphrase
        (check-readable
         (->EncryptedLocalFileSecretsStorage path passphrase))
        (gpg-agent/with-passphrase path message passphrase
          (check-readable
           (->EncryptedLocalFileSecretsStorage path passphrase)))))))

(defn resource-secrets
  "Get secrets from a resource, either in 'secrets' or with a given name."
  [& {:keys [env job-name path passphrase] :as args}]
  (let [[path resource] (->> (apply-map valid-paths args)
                             (map (juxt identity io/resource))
                             (filter second)
                             (first))
        message (format "Passphrase for resource %s." path)]
    (when resource
      (if passphrase
        (check-readable
         (->EncryptedResourceStorage resource passphrase))
        (gpg-agent/with-passphrase (str "resource-" path) message passphrase
          (check-readable
           (->EncryptedResourceStorage resource passphrase)))))))

(defn create-secrets-storage [env & {:keys [passphrase path]}]
  (->EncryptedLocalFileSecretsStorage
   (or path (format "secrets/%s-secrets.edn.gpg" (name env)))
   passphrase))

