(ns circle-util.retry
  (:require [clj-time.core :as time]
            [clojure.core.typed :as t])
  (:use [circle-util.except :only (throw-if-not throwf)]
        [circle-util.time :only (from-now to-millis)]
        [circle-util.seq :only (any?)]
        [slingshot.slingshot :only (try+ throw+)])
  (:import org.joda.time.Period))

(t/warn-on-unannotated-vars)

(defn period? [x]
  (instance? org.joda.time.ReadablePeriod x))

(defn parse-args [args]
  (if (map? (first args))
    {:options (first args)
     :f (second args)}
    {:f (first args)}))

(defn catch-dispatch [options throwable]
  (let [{:keys [catch]} options
        throwable? (instance? Throwable throwable)
        caught-map? (map? throwable)]
    (cond
     (nil? catch) :default
     (and throwable? (sequential? catch) (seq catch) (every? #(isa? % Throwable) catch)) :seq-throwables
     (and caught-map? (keyword? catch)) :slingshot-keyword
     (and caught-map? (vector? catch)) :slingshot-vector
     (fn? catch) :fn)))

(defmulti catch? "decides whether this exception should be caught" catch-dispatch)

(defmethod catch? :default [options throwable]
  false)

(defmethod catch? :seq-throwables [options throwable]
  (let [exceptions (:catch options)]
    (any? (fn [e]
            (instance? e throwable)) exceptions)))

(defmethod catch? :slingshot-keyword [options throwable]
  (get throwable (:catch options)))

(defmethod catch? :slingshot-vector [options throwable]
  (let [v (:catch options)]
    (= (get throwable (first v)) (second v))))

(defmethod catch? :fn [options throwable]
  ((:catch options) throwable))

(defn retry? [options]
  (let [{:keys [end-time tries]} options]
    (cond
     (time/after? (time/now) end-time) false
     (and (integer? tries) (<= tries 1)) false
     :else true)))

(defn success? [options result]
  (let [success-fn (-> options :success-fn)]
    (cond
     (= success-fn :no-throw) true
     success-fn (success-fn result)
     (and result (not success-fn)) true
     :else false)))

(defn fail
  "stuff to do when an iteration fails. Returns new options"
  [options]
  (when (-> options :sleep)
    (Thread/sleep (-> options :sleep (from-now) (to-millis))))
  (update-in options [:tries] (fn [tries]
                                (if (integer? tries)
                                  (dec tries)
                                  tries))))

(defn wait-for* [{:keys [options f]}]
  (let [timeout (-> options :timeout)]
    (try+
      (let [result (f)]
        (if (success? options result)
          result
          (if (retry? options)
            #(wait-for* {:options (fail options)
                         :f f})
            (if (:throw-on-timeout? options)
              (throwf "failed to become ready")
              result))))
      (catch Object t
        (when-let [hook (-> options :error-hook)]
          (hook t))
        (if (and (catch? options t) (retry? options))
          #(wait-for* {:options (fail options)
                       :f f})
          (throw+))))))

(t/ann ^:no-check wait-for 
       (t/All [x]
         (t/IFn
           [(t/IFn [-> x]) -> x]
           [(t/HMap :optional {:tries t/Int
                               :timeout Period
                               :sleep Period})
            (t/IFn [-> x]) 
            -> x])))
(defn wait-for
  "Like robert bruce, but waits for arbitrary results rather than just
  exceptions.

 - f - a fn of no arguments.

 Options:

 - sleep: how long to sleep between retries, as a joda
   period. Defaults to 1s.

 - tries: how many times to retry before throwing. Defaults to 10 (or
   unlimited if timeout is given)

 - timeout: a joda period. Stop retrying when period has elapsed,
   regardless of how many tries are left.

 - catch: Can be one of several things:
     - a seq of exception classes to catch and retry on
     - an fn of one argument, the thrown exception
     - if the exception is a slingshot throwing a map, can be a
       keyword, or a vector of a key and value, destructuring
       slingshot-style.

   If the exception matches the catch clause, wait-for
   retries. Otherwise, the error-hook is called, and then the
   exception is thrown.

 - success-fn: a fn of one argument, the return value of f. Stop
   retrying if success-fn returns truthy. If not specified, wait-for
   returns when f returns truthy. May pass :no-throw here, which will
   return truthy when the f doesn't throw.

 - error-hook: a fn of one argument, an exception. Called every time
   fn throws, before the catch clause decides what to do with the
   exception

 - throw-on-timeout?: if true, throw when when the fn fails to return
   successfully. Doesn't catch exceptions thrown by the fn, only
   controls the final throw when retries are exhausted. Defaults to
   true."

  {:arglists
   '([fn] [options fn])}
  [& args]
  (let [{:keys [options f] :as parsed-args} (parse-args args)
        {:keys [success-fn timeout tries sleep throw-on-timeout?]
         :or {sleep (time/seconds 1)
              throw-on-timeout? true}} options
         tries (if (and sleep timeout (not (-> options :tries)))
                :unlimited
                (or (-> options :tries) 10))
        _ (when sleep
            (throw-if-not (period? sleep) "sleep must be a period"))
        end-time (when timeout
                   (throw-if-not (period? timeout) "timeout must be a joda period")
                   (time/plus (time/now) timeout))
        options (-> options
                    (assoc :end-time end-time
                           :tries tries
                           :sleep sleep
                           :throw-on-timeout? throw-on-timeout?))]
    (throw-if-not (-> parsed-args :f fn?) "couldn't find fn")
    (throw-if-not (or (= :no-throw success-fn)
                      (fn? success-fn)
                      (nil? success-fn)) "success-fn must be a fn")

    (trampoline #(wait-for* {:options options :f f}))))
