(ns circle-util.re
  (:require [clojure.core.typed :as t :refer (Option)]
            [circle-util.timing :refer (with-time-limited)]
            [circle-util.type.ann]
            [circle-util.map :refer (filter-vals)]
            [circle-util.seq :refer (any?)]
            [circle-util.string :as c-str])
  (:import java.util.regex.Pattern))

(t/warn-on-unannotated-vars)

(t/ann flag-keys (t/Map t/Keyword t/Int))
(def flag-keys
  {:insensitive Pattern/CASE_INSENSITIVE
   :literal Pattern/LITERAL})

(t/ann flags->bitmask [(t/Map t/Keyword Boolean) -> t/Int])
;; core.typed currently chokes in (filter vals) -> (keys)
(defn flags->bitmask
  [flags]
  (->> flags
       (filter-vals identity)
       (keys)
       (map (fn [k] (get flag-keys k 0)))
       (reduce bit-or 0)))

(t/ann re [String 
           & :optional {:literal Boolean
                        :insensitive Boolean}
           -> Pattern])
(defn re
  "Creates a regex from a string"
  [s & {:keys [insensitive literal]
        :or {insensitive false
             literal false}}]
  (java.util.regex.Pattern/compile s (int (flags->bitmask {:insensitive insensitive
                                                           :literal literal}))))

(t/ann re-> [String java.util.regex.Pattern -> (t/U nil String (t/Vec (Option String)))])
(defn re->
  "re-find, with arguments reversed, for arrowing"
  [s re]
  (re-find re s))

(defn re-seq-> [s re]
  (re-seq re s))

(defn safe-re-matches
  "re-matches that can be used against untrusted input with timeout.  Returns nil on timeout"
  [re s & {:keys [timeout-ms] :or {timeout-ms 2000}}]
  (with-time-limited timeout-ms nil (re-matches re s)))

(defn regex-matches?
  "Compares pattern to s in simple test.  If pattern is a regex in form /.../ a regex test is used too"
  [pattern s]
  (or (= pattern s)
      (try
        (and (-> pattern count (> 2))
             (-> pattern first (= \/))
             (-> pattern last (= \/))
             (-> pattern
                 (subs 1 (- (count pattern) 1))
                 re-pattern
                 (safe-re-matches s)
                 boolean))
        (catch Exception e
          false))))

(defn any-regex-matches?
  "Like regex-matches? but pattern can be a collection.  If pattern is a collection, then True if any of them matches the regex test"
  [patterns s]
  (cond
   (string? patterns) (regex-matches? patterns s)
   (coll? patterns) (any? #(regex-matches? % s) patterns)))

(defn anchor
  "Returns a new regex, fully anchored, containing ^ and $ chars"
  [pattern]
  (let [pattern (if (c-str/starts-with? (str pattern) "^")
                  pattern
                  (re (str "^" pattern)))
        pattern (if (c-str/ends-with? (str pattern) "$")
                  pattern
                  (re (str pattern "$")))]
    pattern))

