(ns circle-util.queue
  (:require [langohr.core :as rmq]
            [langohr.basic :as rmq-basic]
            [langohr.channel :as rmq-channel]
            [langohr.consumers :as rmq-consumers]
            [langohr.exchange :as rmq-exchange]
            [langohr.http :as rmq-http]
            [langohr.queue :as rmq-queue]
            [cheshire.core :as json]
            [clojure.tools.logging :as log]
            [clojure.core.async :as async]
            [circle-util.except :refer (eat)]
            [circle-util.aws.cloudwatch :as cloudwatch]
            [clj-statsd :as statsd]
            [clj-time.coerce :as time-coerce])
  (:import [java.util.Date]
           [com.rabbitmq.client Connection
                                Channel]))

(defprotocol CloseableRabbitMQObject
  (close [x]))

(extend-protocol CloseableRabbitMQObject
  ;; Eat exceptions since there is no possible action that we can take when
  ;; disconnect fails.
  ;; http://blogs.msdn.com/b/oldnewthing/archive/2008/01/07/7011066.aspx
  Connection
    (close [connection] (eat (rmq/close connection)))
  Channel
    (close [chan] (eat (rmq-channel/close chan))))


(defn connect
  "Get a RabbitMQ connection"
  [uri]
  (rmq/connect {:uri uri}))

(defn open-channel
  "Open a new channel on a connection"
  [connection]
  (rmq-channel/open connection))

(defn with-channel
  "Open a new channel on connection and pass it to f.
  The channel is disposed of once f has returned or thrown an Exception."
  [connection f]
  (let [chan (open-channel connection)]
    (try
      (f chan)
      (finally
        (close chan)))))

(defn ensure-queue-exists
  "Ensure that a given queue name exists on the given channel."
  [channel queue-name]
  (rmq-exchange/declare channel
                        queue-name
                        "direct"
                        {:durable true})
  (rmq-queue/declare channel
                     queue-name
                     {:auto-delete false
                      :durable true
                      :exclusive false})
  (rmq-queue/bind channel
                  queue-name
                  queue-name
                  {:routing-key queue-name}))


(defn- clone-channel
  "Get another channel on the same connection as a given channel."
  [channel]
  (open-channel (.getConnection channel)))

(defn submit
  "Send a message to a RabbitMQ queue."
  [queue message channel]
  (rmq-basic/publish channel queue queue message {:timestamp  (java.util.Date.)
                                                  :content-type "application/json"}))

(defn ack
  "Middleware that acks a message once the client returns without throwing, or
  rejects it if the client throws."
  [f]
  (fn [ch meta data & args]
    (try (let [result (apply f ch  meta data  args)]
           (rmq-basic/ack ch (:delivery-tag meta))
           result)
         (catch Throwable e
           (rmq-basic/reject ch (:delivery-tag meta) false)
           (throw e)))))

(defn consume-json
  "Middleware that JSON-encodes the data a client will receive."
  [f]
  (fn [ch meta ^bytes data & args]
    (apply f ch meta (json/decode (String. data "utf-8") true) args)))

(defn return-json
  "Middleware that JSON-encodes the return value of a function."
  [f]
  (fn [& args]
    (json/encode (apply f args))))

(defn publish-return
  "Middleware that publishes the results of a client to a specific queue."
  [queue f]
  (fn [ch & args]
    (let [result (apply f ch args)]
      ;; TODO: are we 100% sure we want to reuse the channel here?
      (submit queue result ch)
      result)))

(defn timing
  "Generic timing middleware."
  [timing-reporting-function f]
  (fn [ch meta & args]
    (let [send-time (-> meta :timestamp time-coerce/to-long)
          receive-time (System/currentTimeMillis)]
      (try
        (apply f ch meta args)
        (finally
          (let [finish-time (System/currentTimeMillis)]
            (try
              (timing-reporting-function :latency (- receive-time send-time)
                                         :processing-time (- finish-time receive-time)
                                         :total-time (- finish-time send-time))
              (catch Exception e
                (statsd/increment :circle-util.queue.timing.error)
                (log/warn e "Exception in reporting timing")))))))))

(defn statsd-timing
  "Middleware that uses statsd to report metrics like queue latency + processing time.
  These will be named circle-util.queue.{latency,processing-time,total-time}. You can
  filter by job-name in datadog in order to get the metrics for a specific service."
  [f]
  (timing (fn [& {:keys [latency processing-time total-time]}]
            (statsd/timing :circle-util.queue.latency latency)
            (statsd/timing :circle-util.queue.processing-time processing-time)
            (statsd/timing :circle-util.queue.total-time total-time))
          f))

(defn cloudwatch-timing
  "Middleware that reports queue metrics to Amazon CloudWatch."
  [f]
  (timing (fn [& {:keys [latency processing-time total-time]}]
            (cloudwatch/put-metric-data {:queue-latency latency
                                         :queue-processing-time processing-time
                                         :queue-total-time total-time}
                                        :unit "Milliseconds"))
          f))

(defn log-errors [f]
  (fn [& args]
    (try (apply f args)
         (catch Throwable e
           (statsd/increment :circle-util.queue.error)
           (log/error e "Exception in queue handler!")))))

(defn create-report-errors-middleware
  "Create a middleware that will report exceptions with the given reporter
  function.
  The reporter function must accept one argument, the exception that was
  thrown.
  circle-util.queue does not provide any reporter implementations."
  [reporter]
  (fn [f]
    (fn [& args]
      (try (apply f args)
           (catch Exception ex
             (reporter ex)
             (throw ex))))))

(def standard-middleware
  (comp log-errors statsd-timing consume-json ack))

(defn subscribe
  "Subscribe to a queue and process its contents using a provided function.
  You may want to wrap your function in middleware before passing it here.
  > (subscribe q f chan)

  Returns a new channel that is connected to the same broker as the given
  channel. To unsubscribe you must use the channel that was returned from
  subscribe, rather than the channel that was passed in to subscribe."
  [queue f channel & {:keys [middleware]
                      :or {middleware standard-middleware}}]
  (let [chan (clone-channel channel)]
    (rmq-consumers/subscribe chan queue (middleware f))
    chan))

(defn unsubscribe
  "Unsubscribe from the queue and close the channel."
  [queue channel]
  (rmq-basic/cancel channel queue)
  (rmq-channel/close channel))

(defn- http-api-settings-from-amqp-uri
  "Given an AMQP URL, derive the HTTP/HTTPS API settings.

   This does the right thing for a local RabbitMQ install and CloudAMQP's
   setup. If you're dealing with some other RabbitMQ install then you probably
   want to explicitly pass protocol and port parameters to get what you actually
   want."
  [amqp-uri & {:keys [protocol port]}]
  (let [settings (rmq/settings-from amqp-uri)
        local-dev-rabbit (not (nil? (re-find #"@localhost:" (:host settings))))]
    (assoc settings
           :http-api (format "%s://%s:%d"
                             (or protocol (if local-dev-rabbit "http" "https"))
                             (:host settings)
                             (or port (if local-dev-rabbit 15672 443))))))

(defn record-queue-size
  "Periodically report the size of the queue in a background thread."
  [uri queue interval metric-send-fn]
  (let [settings (http-api-settings-from-amqp-uri uri)]
    (async/go
      (rmq-http/connect! (:http-api settings) (:username settings) (:password settings))
      (while true
        (try
          (let [queue-stats (rmq-http/get-queue (:vhost settings) queue)]
            (metric-send-fn (:messages queue-stats)))
          (catch Exception e
            (statsd/increment :circle-util.queue.record-queue-size.fail)
            (log/errorf e "Exception while recording queue size: %s" e)))
        (Thread/sleep (* 1000 interval))))))
