(ns circle-util.middleware
  (:refer-clojure :exclude (remove compile))
  (:require [circle-util.except :refer (throw-if-not)]
            [circle-util.seq :as seq]))

(defn middleware?
  "a middleware is either a defn or a vector of [keyword anonymous-fn]."
  [m]
  (or (var? m) (and (vector? m)
                    (= 2 (count m))
                    (keyword? (first m))
                    (fn? (second m)))))

(defn assert-middleware! [m]
  (throw-if-not (middleware? m) "not a valid middleware: %s" m))

(defn wrap-inner
  "Adds a middleware to the stack. The first middleware will be the inner-most fn run"
  [stack & vs]
  (doseq [v vs]
    (assert-middleware! v))
  (apply conj stack vs))

(defn wrap-outer
  "Adds a middleware to the stack. The first middleware will be the inner-most fn run"
  [stack & vs]
  (doseq [v vs]
    (assert-middleware! v))
  (seq/concatv (apply vector vs) stack))

(defn middleware-key [m]
  (if (var? m)
    m
    (first m)))

(defn middleware-fn [m]
  (assert-middleware! m)
  (if (var? m)
    m
    (second m)))

(defn remove
  "Remove a middleware from the stack. Takes a middleware key"
  [stack m]
  (vec (seq/remove= (middleware-key m) stack)))

(defn insert-after [stack old new]
  (vec (seq/insert-after #(= % (middleware-key old)) (middleware-key new) stack)))

(defn insert-before [stack old new]
  (vec (seq/insert-before #(= % (middleware-key old)) (middleware-key new) stack)))

(defn compile [stack kernel]
  (reduce (fn [final m]
            ((middleware-fn m) final)) kernel stack))

;; TODO: compile-with-timing

(defn create-stack
  "returns a stack of middleware.

   middleware are First-In-Last-Out order. i.e. (create-stack #'foo #'bar) the compiled fn will call foo, then bar, then the kernel fn"
  [& ms]
  (apply wrap-inner [] (reverse ms)))

(defn with-stack
  "Takes f, a fn. Wraps f in all middleware, then calls it. Every fn
  in stack must have compatible signatures."
  [stack f & args]
  (apply (compile stack f) args))
