(ns circle-util.map
  "Fns for working with maps"
  (:require [clojure.core.typed :as t])
  (:import clojure.lang.IMapEntry))

(t/warn-on-unannotated-vars)

(defn has-keys?
  "True if m contains all keys listed"
  [m keyseq]
  (every? (fn [k]
            (contains? m k)) keyseq))

(defn map-keys
  "Calls f, a fn of one argument on each key in m. Replaces each key in m with (f k)"
  [f m]
  (->> m
       (map (fn [pair]
              [(f (key pair)) (val pair)]))
       (into {})))

(defn map-vals
  "Calls f, a fn of one arg on each value in m. Returns a new map"
  [f m]
  (->> m
       (map (fn [pair]
              [(key pair) (f (val pair))]))
       (into {})))

(t/ann filter-pairs (t/All [x y]
                         [[(IMapEntry x y) -> t/Any] (t/Map x y) -> (t/Map x y)]))
(defn filter-pairs
  "Calls f, a fn of one arg on each pair in m. Returns a new map with all pairs
  that returned truthy"
  [f m]
  (->> (seq m)
       (filter (t/fn [pair :- (IMapEntry x y)]
                 (f pair)))
       (into {})))

(t/ann filter-vals (t/All [y] [[y -> t/Any] (t/Map t/Any y) -> (t/Map t/Any y)]))
(defn filter-vals
  "Calls f, a fn of one arg on each value in m. Returns a new map with all the values that returned truthy"
  [f m]
  (filter-pairs (t/fn [pair :- (IMapEntry t/Any y)]
                  (f (val pair))) m))

(defn remove-vals
  "Calls f a fn of one arg on each map value in m. Returns a new map for missing
  all values that returned truthy"
  [f m]
  (filter-vals #(not (f %)) m))

(defn rename-keys
  "replace is a map of old keys to new keys. Replaces keys in m with new versions"
  [replace m]
  (map-keys (fn [k]
              (or (get replace k) k)) m))

(defn invert
  "Swap keys and values"
  [m]
  (zipmap (vals m) (keys m)))

(defn submap?
  "True if every key and value in m1 is contained in m2"
  [m1 m2]
  (let [k (keys m1)]
    (= (select-keys m2 k)
       m1)))

(defn contains-in?
  "True if the nested associative structure m contains a value picked out by
  the sequence of keys [k & ks]."
  [m [k & ks]]
  (if ks
    (let [sentinel (Object.)
          value (get-in m (cons k ks) sentinel)]
      (not= value sentinel))
    (contains? m k)))
