(ns circle-util.gpg-agent
  (:require [circle-util.core :refer (apply-map)]
            [circle-util.sh :as sh]
            [cemerick.url :as url]
            [clojure.string :as str]
            [slingshot.slingshot :refer :all]))

(defn gpg-agent
  "Run a command against the running gpg-agent."
  [format-string & args]
  (let [;; nil prints as NULL, and empty strings break the input.
        sanitize #(url/url-encode (or % " "))
        command (apply format format-string (map sanitize args))
        results (sh/sh! (sh/q (gpg-connect-agent)) :in command)]
    (if-let [[_ code message] (re-find #"^ERR (\d+) (.+)$" (:out results))]
      ;; if there's an error section, throw.
      (throw+ {:gpg-agent-error code
               :message message})
      ;; return the data section. this may be nil, that's fine.
      (let [[_ data] (re-find #"(?m)^D (.*)$" (:out results))]
        (-> (or data "")
            ;; url/url-decode treats + as space
            (str/replace #"\+" "%2B")
            (url/url-decode))))))

(defn clear-passphrase
  "gpg-agent caches passphrases. Sometimes we need to clear it if e.g. decryption fails."
  [cache-id]
  (gpg-agent "CLEAR_PASSPHRASE %s\n" (hash cache-id)))

(defn get-passphrase
  "Prompt for a passphrase with gpg-agent."
  [cache-id & {:keys [error-message prompt description]
               :or {error-message " "
                    prompt "Passphrase"
                    description "Enter your passphrase."}}]
  (gpg-agent "GET_PASSPHRASE --data %s %s %s %s\n" (hash cache-id) error-message prompt description))

(defmacro with-passphrase
  "Request a passphrase from gpg-agent, and retry this body once if the passphrase is wrong."
  [cache-id description name & body]
  `(let [description# ~description
        cache-id# ~cache-id
        ~name (get-passphrase cache-id# :description description#)]
    (try+ (do ~@body)
          ;; if it tells us the passphrase was wrong, clear the cache and try again.
          (catch :incorrect-passphrase ex#
            (clear-passphrase cache-id#)
            (let [~name (get-passphrase cache-id#
                                        :description description#
                                        :error-message "Incorrect passphrase; try again.")]
              ;; if the passphrase is wrong *again*, clear the cache and rethrow.
              (try+ (do ~@body)
                    (catch :incorrect-passphrase ex#
                      (clear-passphrase cache-id#)
                      (throw+ ex#))))))))

(defn input
  "One-time use input. Won't be cached, won't be verified. Useful for prompts for sensitive
  data, like AWS keys, to store and use later."
  [& {:as opts}]
  (let [name (java.util.UUID/randomUUID)]
    (try (apply-map get-passphrase name opts)
         (finally
           (clear-passphrase name)))))
