(ns circle-util.fs
  "utilities for working with files and directories"
  (:require [clojure.set :as set]
            [clojure.core.typed :as t]
            [clojure.java.io :as io]
            [clojure.core.strint :refer (<<)]
            fs
            [pathetic.core :as pathetic]
            [circle-util.base64 :as base64]
            [circle-util.seq :as seq]
            [circle-util.core :refer (apply-map)]
            [circle-util.string :as str]
            [circle-util.except :refer (throw-if-not)]
            [circle-util.sh :as sh]
            [frinj.calc :as frinj])
  (:import circle_util.RubyDir
           (clojure.lang Keyword
                         IPersistentMap)
           com.google.common.hash.Hashing
           frinj.core.fjv
           (java.io ByteArrayOutputStream
                    FileInputStream
                    FileOutputStream
                    File
                    InputStream
                    OutputStream
                    IOException)
           (java.nio.file attribute.PosixFileAttributeView
                          attribute.PosixFilePermissions
                          FileSystems
                          FileVisitResult
                          FileVisitor
                          Files
                          LinkOption
                          Path)
           java.text.DecimalFormat))

(t/warn-on-unannotated-vars)

(t/ann frinj.calc/frinj-init! [-> t/Any])
(frinj/frinj-init!) ;; TODO: this sucks

(t/ann ^:no-check normalize-path [String -> String])
(defn normalize-path
  "Removes any removable same-dir and parent-dir references in 'path'. Also
  removes any leading or trailing whitespace.
  E.g.
  \"foo//bar\" -> \"foo/bar\"
  \"foo/bar/../baz\" -> \"foo/baz\""
  [path]
  (pathetic/normalize path))

(t/ann ^:no-check path [String -> Path])
(defn path
  "Given a string representing a path, return a java.nio.file.Path"
  [path]
  (.getPath (FileSystems/getDefault) path (into-array String [])))

(t/ann ^:no-check all-files [String & :optional {:recursive? Boolean
                                                 :max-depth Integer
                                                 :only-one? Boolean
                                                 :matcher-fn (t/IFn [String -> Boolean])} -> (t/Seqable String)])
(defn all-files
  "Returns a list of all files in the repos"
  [dir & {:keys [recursive? max-depth only-one? matcher-fn]
          :or {recursive? true max-depth Integer/MAX_VALUE only-one? false matcher-fn (constantly true)}}]
  (when (and dir
             (fs/exists? dir)
             (fs/directory? dir))
    (let [result (transient [])
          visitor (reify FileVisitor
                    (preVisitDirectory [this dir attrs]
                      (if (-> dir .getFileName .toString (str/starts-with? "."))
                        FileVisitResult/SKIP_SUBTREE
                        FileVisitResult/CONTINUE))
                    (visitFileFailed [this file exc]
                      FileVisitResult/CONTINUE)
                    (postVisitDirectory [this file exc]
                      FileVisitResult/CONTINUE)
                    (visitFile [this file attrs]
                      (let [matched (and (-> attrs .isDirectory not)
                                         (-> file str matcher-fn))]
                        (when matched
                          (conj! result (str file)))
                        (if (and matched only-one?)
                          FileVisitResult/TERMINATE
                          FileVisitResult/CONTINUE))))]
      (Files/walkFileTree (path dir)
                          (java.util.HashSet.)
                          (if recursive? max-depth 1)
                          visitor)
      (persistent! result))))

;; TODO: slurp can take a union of types: file, string, url, etc..
(t/ann ^:no-check slurp-limit [String & :optional {:limit Long
                                                   :encoding String}
                               -> String])
(defn slurp-limit
  "Same as clojure.core/slurp, but takes an optional argument, :limit, the maximum number of bytes to take"
  [f & opts]
  (let [opts (#'clojure.core/normalize-slurp-opts opts)
        {:keys [limit]} opts]
    (if limit
      (with-open [#^java.io.Reader r (apply io/reader f opts)]
        (let [buf (char-array limit)
              len (.read r buf)]
          (if (pos? len)
            (String. buf 0 len)
            "")))
      (apply slurp f opts))))

(defn slurp-binary
  [x]
  (let [out-stream (ByteArrayOutputStream.)]
    (with-open [stream (io/input-stream x)]
      (io/copy stream out-stream))
    (.toByteArray out-stream)))

(defn files-matching
  "Returns a list of files that match re."
  [dir re & {:as opts}]
  (let [opts (merge {:matcher-fn #(re-find re %)} opts)]
    (apply-map all-files dir opts)))

(defn dir-contains-files?
  "True if the directory contains files matching any of the regexes"
  [dir & regexes]
  (let [regexes (or (seq regexes) [#".*"])
        matcher-fn (fn [file] (some #(re-find % file) regexes))
        opts {:matcher-fn matcher-fn :only-one? true}]
    (-> (apply-map all-files dir opts)
        seq
        boolean)))

(defn repo-dir-contains-content?
  "Find files in dir whose name matches file-regex, and search through matching
  files for content matching content-regex.
  As an optimization, this function used `git-grep`, and therefore the dir must
  be located inside a git repsository.
  Returns true if there are any files that the match, and false otherwise.
  Throws an IOException if there was an error running the search."
  [dir & {:keys [file-pattern content-regex]}]
    (let [result (sh/sh (sh/q-chain (cd ~(sh/q-arg dir))
                                    (git grep
                                         --cached
                                         --name-only
                                         --quiet
                                         -e ~(sh/q-arg content-regex)
                                         -- ~(sh/q-arg file-pattern))))]
      (case (:exit result)
        0 true
        1 false
        (throw (IOException. ;; git-grep returns 128 on failure
                 (format "Error checking repository %s (%s) for content %s: %s"
                         dir file-pattern content-regex result))))))

(defn re-file
  "Returns the contents of the file that match regex"
  [file re]
  (try
    (when (and file (fs/exists? file))
      (re-find re (slurp-limit file :limit 200000)))
    (catch java.io.FileNotFoundException e
      nil)))

(defn re-file?
  "True if the contents of the file match the regex"
  [file re]
  (boolean
    (seq (re-file file re))))

(defn line-count
  "returns the number of lines in the file"
  [file]
  (if (and file (fs/exists? file))
    (->> file (slurp) (re-seq #"\n") (count))
    0))

(defn move [a b]
  (Files/move (path a) (path b) (into-array java.nio.file.CopyOption [])))

(t/ann tmp-spit [String -> String])
(defn tmp-spit
  "Creates a temp file, spits content to it, and returns the path to the temp file"
  [content]
  (let [f (fs/tempfile)]
    (spit f content)
    f))

(t/ann sudo-spit [String String -> nil])
(defn sudo-spit
  [file content]
  (let [path (tmp-spit content)]
    (sh/sh! (sh/q (sudo -n mv ~path ~file)))
    nil))

(t/ann sudo-exists? [String -> Boolean])
(defn sudo-exists?
  [path]
  (-> (sh/shq (sudo -n -- test -e ~(sh/q-arg path)))
      :exit
      zero?))

(defn ^InputStream in-stream
  "Given a path, return an input stream for reading. You probably want to use with-open on this."
  [^String path]
  (FileInputStream. (File. path)))

(defn ^OutputStream out-stream
  "Given a path, return an output stream for writing to the file. You probably want with-open on this"
  [^String path]
  (FileOutputStream. (File. path)))

(t/ann split [String -> (t/Vec String)])
(defn split
  "fs/split without using reflection"
  [^String path]
  (into [] (.split #"/" path 0)))

(t/ann ^:no-check relative [String String -> String])
(defn relative
  "Given a and b are two paths, and b is contained in a, return b relative to a"
  [a b]
  (-> (seq/drop-duplicates (split a)
                           (split b))
      (second)
      (#(apply fs/join %))))

(defn drop-dir
  "given a path, drop the first dir off"
  [d]
  (->> d
       (split)
       (drop 1)
       (apply fs/join)))

(defn drop-extension
  "Given a path, drop the extension (.clj) off the filename"
  [^String path]
  (-> path
      (.split "\\.")
      (into [])
      (butlast)
      (#(apply fs/join %))))

(defn drop-filename
  "Given a path, drop the filename and return the containing path
   e.g. /path/to/file.txt -> /path/to"
  [^String path]
  (-> path
      (split)
      (into [])
      (butlast)
      (#(apply fs/join %))))

(defn md5
  "Returns a com.google.common.hash.HashCode for the MD5 hash of a file's
   contents"
  [^String path]
  (-> path (File.) (com.google.common.io.Files/hash (Hashing/md5))))

(t/ann ^:no-check md5-base64 [String -> String])
(defn md5-base64
  "returns the md5 of the file contents"
  [^String path]
  (-> path md5 .asBytes base64/encode))

(defn md5-hex
  "The hexadecimal encoded MD5 hash of a file's contents."
  [^String path]
  (-> path md5 .toString))

(defmacro with-delete
  "Delete path after the body is done"
  [path & body]
  `(try
     ~@body
     (finally
      (when ~path
        (fs/delete ~path)))))

(defmacro with-delete-dir
  "Delete path of dir after the body is done"
  [path & body]
  `(try
     ~@body
     (finally
      (when ~path
        (fs/deltree ~path)))))

(defn ->attribute-view
  [path]
  (let [path (io/file path)]
    (Files/getFileAttributeView (.toPath path)
                                PosixFileAttributeView
                                (into-array LinkOption []))))

(t/ann ^:no-check set-mode [String String -> t/Any])
(defn set-mode
  "Set a file's mode using a mode-string made up of three groups of rwx for
  user, group and other.
  If a permission is set to '-' that permission is removed.
  E.g.
  rw-rw---- is equivalent to 660 - user has read+write, group has read+write,
  other has no access."
  [mode-string path]
  (let [mode (PosixFilePermissions/fromString mode-string)
        attr-view (->attribute-view path)]
    (.setPermissions attr-view mode)))

(t/ann ^:no-check get-mode [String -> String])
(defn get-mode
  "Get a file's mode as a mode-string (see set-mode)."
  [path]
  (-> path
      ->attribute-view
      .readAttributes
      .permissions
      PosixFilePermissions/toString))

(defmacro with-temp-file
  "creates a temp file, binds local-name to the path of the temp
  file. runs body and deletes the file"
  [local-name & body]
  `(let [~local-name ^String (fs/tempfile)]
     (try
       (set-mode "rw-------" ~local-name)
       ~@body
       (finally
        (fs/delete ~local-name)))))

(defmacro with-temp-dir
  "creates a temp directory, binds local-name to the path of the temp
  file. runs body and removes the directory"
  [dir-name & body]
  `(let [~dir-name (fs/tempdir)]
     (try
       ~@body
       (finally
        (fs/deltree ~dir-name)))))

(t/ann ^:no-check glob-max-depth [(t/U String (t/Seqable String)) (t/Option Boolean) -> Long])
(defn glob-max-depth
  [glob ruby-glob]
  (cond
   ruby-glob Integer/MAX_VALUE
   (coll? glob) (->> glob
                    (map #(glob-max-depth % ruby-glob))
                    (apply max))
   (str/contains? "**" glob) Integer/MAX_VALUE
   (str/contains? "/" glob) (->> glob (seq) (filter #{\/}) count (+ 1))
   :else 1))

(t/ann ^:no-check glob-matcher-fn [String (t/U String (t/Seqable String)) (t/Option Boolean) -> [String -> Boolean]])
(defn glob-matcher-fn
  [dir glob ruby-glob]
  (if (coll? glob)
    (let [matcher-fns (map #(glob-matcher-fn dir % ruby-glob) glob)]
      (fn [file] (some #(% file) matcher-fns)))
    (let [pattern (if ruby-glob
                    (format "regex:%s" (RubyDir/convertGlobToRegEx glob))
                    (format "glob:%s" glob))
          matcher (.getPathMatcher (FileSystems/getDefault) pattern)
          matcher-fn (fn [file] (.matches matcher (path (relative dir file))))]
      matcher-fn)))

(t/ann ^:no-check glob [String (t/U String (t/Seqable String)) & :optional {:ruby-glob Boolean} -> (t/Seq String)])
(defn glob
  "Returns all files that match glob, under path. Returns paths relative to dir"
  [dir glob & {:keys [ruby-glob] :as opts}]
  (let [matcher-fn (glob-matcher-fn dir glob ruby-glob)
        max-depth (glob-max-depth glob ruby-glob)
        opts (merge {:matcher-fn matcher-fn :max-depth max-depth} opts)]
    (->> (apply-map all-files dir opts)
         (map (t/fn [file :- String] (relative dir file))))))

(t/ann disk-map (IPersistentMap String Keyword))
(def disk-map
  "Table of disk suffixes to frinj units"
  {"K" :kibibytes
   "M" :mebibytes
   "G" :gibibytes})

(defn parse-size
  "Takes a string that represents a disk size, like '5.6G', and returns the value in bytes."
  [in-str]
  (let [[_ number dimension] (re-find #"^([0-9\.]+)([KMG]{1})?i?B?$" in-str)
        number (Double/parseDouble number)]
    (frinj/fj number (get disk-map dimension :bytes))))

(defn ->bytes
  [in-str]
  (-> (parse-size in-str)
      (frinj/to :bytes)
      :v))

(defn ->kbytes
  [in-str]
  (-> (parse-size in-str)
      (frinj/to :kibibytes)
      :v))

(t/ann round [Number -> String])
(defn round [n]
  (->
   (DecimalFormat. "#.##")
   (.format n)))

(t/ann ^:no-check emit-size [fjv & :optional {:output-unit Keyword} -> String])
(defn emit-size
  "Takes a frinj number of bytes, returns a disk size. Output unit should be recognized by frinj"
  [num & {:keys [output-unit]
          :or {output-unit :gibibytes}}]
  (let [number (-> num (frinj/to output-unit) :v round)
        suffix (-> disk-map set/map-invert output-unit)]
    (<< "~{number}~{suffix}")))

(defn mkdir!
  "Try to create a directory locally. Throws on failure"
  [path]
  (throw-if-not (.mkdir (io/as-file path)) (<< "Failed to create ~{path}"))
  path)

(defn rsync
  [from to]
  (sh/sh!
   (sh/q
    (rsync -aq --delete ~from ~to))))

(t/ann ^:no-check depth-count [String -> Long])
(defn depth-count
  "Returns the number of levels the file is in.  Expects a normalized path"
  [^String path]
  ;; This is an optimized version to be used for i-fs/all-files
  (let [separator java.io.File/separatorChar]
    (loop [r 0
           i (- (.length path) 1)]
      (if (< i 0)
        r
        (recur (if (= (.charAt path i) separator) (+ r 1) r)
               (- i 1))))))

(defn relative-root-dirname
  "Expects a relative path, e.g. rspec/my-subdir/junit.xml and returns
   the highest directory in the tree (or nil).
   Absolute paths, e.g. /tmp/file.txt, will always return /"
  [^String path]
  (when-let [start-dir (fs/dirname path)]
    (loop [dir start-dir]
      (if-let [parent (fs/dirname dir)]
        (recur parent)
        dir))))
