(ns circle-util.datadog
  (:require [clojure.set]
            [clojure.tools.logging :refer (infof errorf)]
            [cheshire.core :as json]
            [clj-http.client :as http]
            [clj-statsd :as statsd]
            [clj-time.core :as time]
            [slingshot.slingshot :refer (try+)]
            [circle-util.capabilities :as capabilities]
            [circle-util.core :refer (defn-once)]
            [circle-util.env :as env]
            [circle-util.time :as c-time]))

(def api-url "https://app.datadoghq.com/api")

;; Should be a map with two keys -- :api_key and :application_key.
(def ^:dynamic
 *datadog-secrets*)

(defn datadog
  "datadog API client"
  [method path & {:keys [body params]
                  :or {params {}}}]
  (if-let [keys *datadog-secrets*]
    (let [params (merge params keys)
          request {:method method
                   :headers {"Content-type" "application/json"}
                   :throw-exceptions true
                   :url (str api-url path "?" (http/generate-query-string params))}
          request (if body (assoc request :body (json/generate-string body)) request)
          response (http/request request)]
      (try
        (json/parse-string (:body response))
        (catch Throwable e
          (throw (ex-info (.getMessage e) {:response response} e)))))
    (do
      (infof "No datadog keys are present - skipping datadog api call")
      nil)))

(defn send-metrics
  "Send coll of time series points, each metric should have points and type,
   as documented at http://docs.datadoghq.com/api/#metrics-post"
  [metrics]
  (let [defaults {:host (capabilities/hostname)
                  :tags [(str "env:" (name (env/env)))]}
        series (map #(merge defaults %) metrics)]
    (datadog :post "/v1/series" :body {:series series})))

(defn send-event
  "Send a datadog event."
  [title text & {:keys [date_happened priority tags alert_type
                        aggregation_key source_type_name]
                 :as optional-args}]
  (let [event (into {:title title
                     :text text} optional-args)]
    (datadog :post "/v1/events" :body event)))

(defn get-tags [hostname]
  (-> (datadog :get (str "/v1/tags/hosts/" hostname)
               :params {:source "users"})
      (get "tags")
      (set)))

(defn add-tags [hostname & tags]
  (datadog :post (str "/v1/tags/hosts/" hostname)
           :body {:tags tags}))

(defn remove-tags [hostname & tags]
  (let [existing-tags (get-tags hostname)
        new-tags (set tags)]
    (datadog :put (str "/v1/tags/hosts/" hostname)
             :body {:tags (vec (clojure.set/difference existing-tags new-tags))})))

(defn query-metrics
  "Query some metrics. E.g.:

    (query-metrics \"max:circle.state.num_masters{env:production}\"
                   (time/minus (time/now) (time/minutes 2)) (time/now))

  Note that this is an unsupported API, and datadog strongly discourages fetching more
  than an hour of data at a time..."
  [query from to]
  (datadog :get "/v1/query"
           :params {:query query
                    :from (/ (clj-time.coerce/to-long from) 1000.0)
                    :to (/ (clj-time.coerce/to-long to) 1000.0)}))

(defmacro with-timing-metric
  "Wrap 'body' with timing instrumentation.
  'metric' is a keyword naming the metrics.
  'tags' is a vector of strings to tag the Datadog metric with.

  Timing metrics are sent to Datadog.

  Custom tags that are prefixed with X- (e.g. X-circle-build) are not sent to
  Datadog. If there are any custom tags then the timing, metric and tags are
  also logged via infof."
  [metric tags & body]
  `(let [tags# ~tags
         datadog-tags# (remove #(re-find #"^[Xx]-" %) tags#)
         start-time# (time/now)]
     (try+
       ~@body
       (finally
         (let [elapsed-time# (c-time/to-millis
                                (c-time/maybe-interval
                                  start-time#
                                  (time/now)))]
           (when (> (count tags#) (count datadog-tags#))
             (infof "with-timing-metric %s %s: took %d milliseconds"
                    ~metric
                    ~tags
                    elapsed-time#))
           (statsd/timing ~metric
                          elapsed-time#
                          {:tags datadog-tags#}))))))

(defmacro locking-with-timing
  "Like with-timing-metric, but to handle the case where you want to use
   (clojure.core/locking ... body ...) and see how much time you spend
   waiting on the lock, vs in the body.

   Always use the metric 'circle.locking' for the lock wait, but tag it
   with the metric in question, and the lock name. This lets us see total
   time on all locks, and also easily get just the time spent for any calls
   waiting on a given lock, or just he particular calls responsible for
   the metric passed in."
  [x metric tags & body]
  `(let [x# ~x
         metric# ~metric
         orig-tags# ~tags
         tags# (conj orig-tags# metric# x#)
         datadog-tags# (remove #(re-find #"^[Xx]-" %) tags#)
         start-time# (time/now)]
     (locking x#
       (let [elapsed-time# (c-time/to-millis (c-time/maybe-interval start-time# (time/now)))]
         (when (> (count tags#) (count datadog-tags#))
           (infof "locking-with-timing circle.locking %s: took %d milliseconds obtaining lock %s"
                  tags# elapsed-time# x#))
         (statsd/timing "circle.locking" elapsed-time# {:tags datadog-tags#})
         (with-timing-metric metric# orig-tags# ~@body)))))

(defn-once init
  (try
    (add-tags (capabilities/hostname) (str "env:" (name (env/env))))
    (catch Exception e
      (errorf e "Can't initialize datadog. So be it.")))
  ;; hopefully datadog statsd is running!
  (statsd/setup "127.0.0.1" 8125))
