(ns circle-util.crypto
  (:require [clojure.core.typed :as t]
            [clj-keyczar.crypt]
            [clj-keyczar.sign]
            [circle-util.base64 :as base64]
            [circle-util.core :refer (byte-array-to-hex-string)])
  (:import javax.crypto.Mac
           javax.crypto.spec.SecretKeySpec))

(t/warn-on-unannotated-vars)

(t/defalias CipherText String)
(t/defalias Signature String)

(t/ann clj-keyczar.crypt/encrypt [String -> CipherText])
(t/ann clj-keyczar.crypt/decrypt [CipherText -> String])
(t/ann clj-keyczar.sign/sign [String -> Signature])
(t/ann clj-keyczar.sign/verify [String Signature -> Boolean])

(defn hmac
  "Generates hmac for string using specified algorithm"
  [^String key ^String s algo-name opts]
  (let [keyspec (SecretKeySpec. (.getBytes key "UTF-8") algo-name)
        mac (Mac/getInstance algo-name)
        _ (.init mac keyspec)
        out-bytes (.doFinal mac (.getBytes s))
        format (or (:out opts) :bytes)]
    (case format
      :bytes out-bytes
      :hex (byte-array-to-hex-string out-bytes)
      :base64 (base64/encode out-bytes))))

(defn hmac-sha256
  "generates hmac-sha256 for the string s using key.

  Optionally takes a `:out` argument to define type: :bytes (default), :hex, :base64"
  [key s & {:as opts}]
  (hmac key s "hmacSHA256" opts))

(defn hmac-sha1
  "generates hmac-sha1 for the string s using key.

  Optionally takes a `:out` argument to define type: :bytes (default), :hex, :base64"
  [key s & {:as opts}]
  (hmac key s "hmacSHA1" opts))
