(ns circle-util.aws.ec2-metadata
  (:import  org.joda.time.DateTime)
  (:require [clojure.tools.logging :refer (errorf)]
            [clj-http.client :as http]
            [clojure.core.typed :as t]
            [clojure.string :as str]
            [circle-util.core :refer (defn-once)]
            [circle-util.except :refer (assert! assert-str!)]
            [circle-util.string :as c-str]
            [circle-util.time :refer (custom-formatters)]))

(t/ann aws-metadata-url String)
(def aws-metadata-url "http://169.254.169.254/latest/meta-data")

(t/ann ^:no-check self-metadata [t/Any -> (t/Option String)])
(defn self-metadata
  "Returns metadata about the current instance. attr is a string/keyword"
  [attr]
  (try
    (let [url (format "%s/%s" aws-metadata-url attr)
          resp (http/get url {:throw-exceptions false
                              :socket-timeout 1000
                              :conn-timeout 1000})]
      (if (= 200 (-> resp :status))
        (-> resp :body)
        (errorf "metadata request to %s returned non-200 code %s, body: %s"
                url (-> resp :status) (-> resp :body))))
    (catch Exception e
      nil)))

(t/ann ^:no-check self-instance-id [-> (t/Option String)])
(defn-once self-instance-id
  "If the local box is an EC2 instance, returns the instance id, else nil."
  (self-metadata "instance-id"))

(t/ann ^:no-check self-instance-type [-> (t/Option String)])
(defn-once self-instance-type
  "If the local box is an EC2 instance, returns the instance type, else nil."
  (self-metadata "instance-type"))

(t/ann ^:no-check self-public-ipv4 [-> (t/Option String)])
(defn-once self-public-ipv4
  (self-metadata "public-ipv4"))

(t/ann ^:no-check self-local-ipv4 [-> (t/Option String)])
(defn-once self-local-ipv4
  (self-metadata "local-ipv4"))

(t/ann ^:no-check self-availability-zone [-> (t/Option String)])
(defn-once self-availability-zone
  (self-metadata "placement/availability-zone"))

(t/ann ^:no-check az-name->region-name [String -> String])
(defn az-name->region-name
  "Takes an AZ name, returns the name of the region the AZ belongs to"
  [az-name]
  (c-str/take az-name (dec (count az-name))))

(t/ann ^:no-check self-region [-> (t/Option String)])
(defn-once self-region
  (some-> (self-availability-zone) az-name->region-name))

(t/ann ^:no-check self-vpc-id [-> (t/Option String)])
(defn-once self-vpc-id
  (let [mac (self-metadata "mac")]
    (self-metadata (format "network/interfaces/macs/%s/vpc-id" mac))))

(t/ann ^:no-check self-subnet-id [-> (t/Option String)])
(defn-once self-subnet-id
  (let [mac (self-metadata "mac")]
    (self-metadata (format "network/interfaces/macs/%s/subnet-id" mac))))

(t/ann ^:no-check self-ami-id [-> (t/Option String)])
(defn-once self-ami-id
  (self-metadata "ami-id"))

(t/ann ^:no-check self-usable-devices [-> (t/Option (t/Seqable String))])
(defn self-usable-devices
  "Returns the devices usable for container bed"
  ;; TODO: Pick a better name
  ;; An instance, will have at least two block devices
  ;;  * ami -- device containing the ami image
  ;;  * root -- gets mapped to /
  ;; The rest should be either ebs or ephemeral devices
  ;; Using a whitelist here, rather than a blacklist just in case
  []
  (letfn [(is-usable? [d] (or (c-str/starts-with? d "ephemeral")
                              (c-str/starts-with? d "ebs")))]
    (some->> (self-metadata "block-device-mapping/")
             (str/split-lines)
             (filter is-usable?)
             (map #(self-metadata (str "block-device-mapping/" %))))))

(t/ann parse-ec2-time [String -> DateTime])
(defn parse-ec2-time [time-str]
  (assert! (clj-time.format/parse (:rfc1123 custom-formatters) time-str)))

(t/ann ^:no-check self-boot-time [-> (t/Option DateTime)])
(defn-once self-boot-time
  "Returns the timestamp when this box was booted"
  ;; Returns the timestamp when this box was booted. Undocumented and unsupported
  ;; http://alestic.com/2009/06/ec2-instance-startup-time-meta-data
  (try
    (let [resp (http/get (format "%s/instance-id" aws-metadata-url) {:throw-exceptions false
                                                                     :socket-timeout 1000
                                                                     :conn-timeout 1000})]
      (when (-> resp :status (= 200))
        (-> resp
            (get-in [:headers "last-modified"])
            (assert-str!)
            (parse-ec2-time))))
    (catch Exception e
      nil)))
