(ns circle-util.aws.cloudwatch
  (:require [circle-util.env :as env]
            [circle-util.except :refer (throw-if)]
            [amazonica.aws.cloudwatch :as cloudwatch]))

(def cloudwatch-metadata (atom {}))

(defn force-string
  [item]
  (if (keyword? item) (name item) (str item)))

(defn metric-dimensions
  "Take a hash-map and turn it into a list of AWS CloudWatch Dimensions."
  [tags]
  (let [default-dims (into {} (filter val {:env (env/env)
                                           :service (:service @cloudwatch-metadata)}))]
    (for [[k v] (merge default-dims tags)]
      {:name (force-string k) :value (force-string v)})))

(defn metric-datum
  "Expand a name and a value into an AWS MetricDatum data-type-alike."
  [metric-name metric-value opts]
  (into {} (filter val
                   {:dimensions (metric-dimensions (:dimensions opts))
                    :metric-name (force-string metric-name)
                    :timestamp (:timestamp opts)
                    :unit (:unit opts)
                    :value metric-value})))

(defn put-metric-data
  "Send some metrics data to AWS CloudWatch."
  [metrics & args]
  (throw-if (nil? (:namespace @cloudwatch-metadata))
            "You must call init and give it a :namespace")
  (let [opts (apply hash-map args)]
    (cloudwatch/put-metric-data :namespace (:namespace @cloudwatch-metadata)
                                :metric-data (map #(metric-datum (key %) (val %) opts) metrics))))

(defn init
  [metadata]
  (swap! cloudwatch-metadata merge metadata))
