(ns chia.static.assets
  (:require [clojure.java.io :as io]
            [hiccup.page :as page]
            [clojure.spec.alpha :as s]
            [clojure.string :as str])
  (:import (java.security MessageDigest)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Asset handling

;; Dynamic vars are used for asset-path options


(def ^:dynamic *content-hashes?*
  "When true, append content-hashes to asset paths."
  false)

(def ^:dynamic *asset-dir*
  "Local directory where assets are written"
  nil)

(def ^:dynamic *asset-host*
  "Remote host where assets are served"
  "")

(defn strip-slash [path]
  (cond-> path
          (str/starts-with? path "/") (subs 1)))

(defn md5
  "Returns md5 hash for string"
  ;; from https://gist.github.com/jizhang/4325757#gistcomment-2196746
  [^String s]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s))]
    (format "%032x" (BigInteger. 1 raw))))

(defn try-slurp [file]
  (try (slurp file)
       (catch Exception e nil)))

(defn asset-file [path]
  (assert *asset-dir* "*asset-dir* must be set")
  (io/file *asset-dir* (strip-slash path)))

(defn read-asset
  "Returns the contents for an asset"
  [path]
  (-> (asset-file path)
      (try-slurp)))

(defn asset-hash [path]
  (some-> (read-asset path)
          (md5)))

(defn asset-path
  "Asset-path function, for use in generating HTML"
  [path]
  (if-not (str/starts-with? path "/")
    path
    (str
     *asset-host*
     path
     (when *content-hashes?*
       (str "?v=" (asset-hash path))))))

(defn write-asset!
  "Write `content` string to an asset file"
  [path content]
  (-> (asset-file path)
      (spit content))
  (println (str " + " path)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; HTML generation

(defn element-tag [kw-or-hiccup]
  (cond-> kw-or-hiccup
          (keyword? kw-or-hiccup) (vector)))

(defn script-tag [str-or-map]
  [:script
   (if (string? str-or-map)
     str-or-map
     (update str-or-map :src asset-path))])

(defn meta-tag [k v]
  [:meta {(if (some #{"Expires"
                      "Pragma"
                      "Cache-Control"
                      "imagetoolbar"
                      "x-dns-prefetch-control"} (name k))
            :http-equiv
            :name) (name k)
          :content v}])

(defn style-tag [str-or-map]
  (if (string? str-or-map)
    [:style str-or-map]
    [:link (-> str-or-map
               (assoc :rel "stylesheet")
               (update :href asset-path))]))

(defn html-page
  "Return HTML string for title and props"
  [{:as props
    :keys [lang
           title
           charset
           styles
           meta
           head
           body]
    body-scripts :scripts/body
    head-scripts :scripts/head
    :or {lang "en"
         charset "UTF-8"}}]
  (page/html5 {:lang lang}

              [:head

               (for [[k v] meta]
                 (meta-tag k v))

               [:meta {:http-equiv "Content-Type"
                       :content (str "text/html; charset=" charset)}]

               (when title
                 [:title title])

               (map style-tag styles)
               (map element-tag head)
               (map script-tag head-scripts)]

              [:body
               (map element-tag body)
               (map script-tag body-scripts)]))

(comment
 (html-page
  {:title "Welcome"
   :styles [{:href "/some/styles.css"}
            ".black {color: #000}"]
   :body [:div#app]
   :scripts/body [{:src "/some/script.js"}
                  "alert('hi!')"]}))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Specs

(s/def ::title string?)
(s/def ::lang string?)
(s/def ::charset string?)

(s/def ::href string?)
(s/def ::style-props (s/keys
                      :req-un [::href]))
(s/def ::styles (s/coll-of (s/or :inline-style string?
                                 :href ::href)))
(s/def ::meta (s/map-of keyword?
                        string?))

(s/def ::src string?)
(s/def ::script-props (s/keys
                       :req-un [::src]))
(s/def ::scripts (s/coll-of (s/or :inline-js string?
                                  :src map?)))
(s/def :scripts/head ::scripts)
(s/def :scripts/body ::scripts)

(s/def ::element (s/or :tag keyword?
                       :element vector?))
(s/def ::body (s/coll-of ::element))

(s/def ::html-page-props
  (s/keys :opt-un [::title
                   ::lang
                   ::charset
                   ::styles
                   ::meta
                   ::body]
          :opt [:scripts/head
                :scripts/body]))

(s/fdef html-page
        :args (s/cat :page-props
                     ::html-page-props)
        :ret string?)