(ns chia.static.assets
  (:require [clojure.java.io :as io]
            [hiccup.page :as page]
            [clojure.spec.alpha :as s]
            [clojure.string :as str])
  (:import (java.security MessageDigest)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Asset handling

;; Dynamic vars are used for asset-path options


(def ^:dynamic *content-hashes?*
  "When true, append content-hashes to asset paths."
  false)

(def ^:dynamic *asset-dir*
  "Local directory where assets are written"
  nil)

(def ^:dynamic *asset-host*
  "Remote host where assets are served"
  "")

(defn strip-slash [path]
  (cond-> path
          (str/starts-with? path "/") (subs 1)))

(defn md5
  "Returns md5 hash for string"
  ;; from https://gist.github.com/jizhang/4325757#gistcomment-2196746
  [^String s]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s))]
    (format "%032x" (BigInteger. 1 raw))))

(defn try-slurp [file]
  (try (slurp file)
       (catch Exception e nil)))

(defn asset-hash [path]
  (assert *asset-dir*)
  (some->> (io/file *asset-dir*
                    (strip-slash path))
           (try-slurp)
           (md5)))

(defn asset-path
  "Asset-path function."
  [path]
  (str

   ;; prefix with asset-host, only for absolute paths
   (when (str/starts-with? path "/")
     *asset-host*)

   path

   (when *content-hashes?*
     (str "?v=" (asset-hash path)))))

(defn write!
  "Write a string to the `output-dir` for `build-id`"
  [path content]
  (assert *asset-dir*)
  (-> (io/file *asset-dir* path)
      (spit content))
  (println (str " + " path)))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; HTML generation

(defn html-page
  "Return HTML string for title and props"
  [title {:as props
          :keys [page/lang
                 meta/charset

                 head/styles

                 body/elements

                 body/scripts
                 body/eval]
          :or {lang "en"
               charset "UTF-8"}}]
  (page/html5 {:lang lang}
              [:head
               [:meta {:charset charset}]

               (when title
                 [:title title])

               (for [href styles]
                 [:link {:rel "stylesheet"
                         :href (asset-path href)}])

               [:body

                (for [element elements]
                  [element])

                (for [script scripts]
                  [:script {:src (asset-path script)}])

                (for [script eval]
                  [:script script])]]))

;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Specs

(s/def :page/lang string?)
(s/def :meta/charset string?)
(s/def :head/styles (s/coll-of string?))
(s/def :body/elements (s/coll-of keyword?))
(s/def :body/scripts (s/coll-of string?))
(s/def :body/eval (s/coll-of string?))

(s/def ::html-page-props
  (s/keys :opt [:page/lang
                :meta/charaset
                :head/styles
                :body/elements
                :body/scripts
                :body/eval]))

(s/fdef html-page
        :args (s/cat :title string? :props ::html-page-props)
        :ret string?)