(ns ch.codesmith.blocks.jdbc
  (:require
    [ch.codesmith.blocks :as cb]
    [clojure.spec.alpha :as s]
    [migratus.core :as migratus]
    [next.jdbc.connection :as conn])
  (:import (com.zaxxer.hikari HikariDataSource)
           (java.sql Connection)
           (javax.sql DataSource)))

;; # Migrations

(defmulti migratus-db-map class)

(defmethod migratus-db-map
  :default [value] value)

(defmethod migratus-db-map
  Connection [conn] {:connection conn})

(defmethod migratus-db-map
  DataSource [ds] {:datasource ds})

(defn migratus-migrate-function [connectable migratus-config]
  (let [config (merge {:db    (migratus-db-map connectable)
                       :store :database}
                      migratus-config)]
    (migratus/init config)
    (migratus/migrate config)))

(defn migrate-db! [ds {:keys [migration-config migration-function] :or {migration-function migratus-migrate-function}}]
  (migration-function ds migration-config))

;; # configuration

(s/def ::jdbcUrl string?)

(s/def ::datasource (s/keys :req-un [::jdbcUrl]))
(s/def ::migration-function (fn [x]
                              (or (fn? x)
                                  (var? x))))
(s/def ::migration-config map?)

(s/def ::configuration (s/keys :req-un [::datasource]
                               :opt-un [::migration-config ::migration-function]))

(def configuration-checker (cb/checker ::configuration))

(def datasource-defaults
  {:autoCommit false})

(defn include-datasource-defaults [config defaults]
  (update config :datasource #(merge defaults %)))

;; # blocks helper

(defn init [config]
  (let [config (configuration-checker (include-datasource-defaults config datasource-defaults))
        ds     (conn/->pool HikariDataSource (:datasource config))]
    (migrate-db! ds config)
    ds))

(defn halt! [_ ^HikariDataSource ds]
  (.close ds))

(defn data-source-config [^HikariDataSource ds]
  {:jdbcUrl              (.getJdbcUrl ds)
   :username             (.getUsername ds)
   :readOnly             (.isReadOnly ds)
   :driverClassName      (.getDriverClassName ds)
   :dataSourceProperties (into {} (.getDataSourceProperties ds))})