(ns ch.codesmith.blocks.jdbc
  (:require
    [ch.codesmith.blocks.core :as cb]
    [clojure.spec.alpha :as s]
    [next.jdbc :as jdbc]
    [next.jdbc.connection :as njc]
    [next.jdbc.protocols :as p])
  (:import (com.zaxxer.hikari HikariDataSource)
           (java.io Closeable)))

(defmethod cb/valid-block?
  :jdbc/datasource
  [_ block]
  (and
    (extends? p/Sourceable block)
    (extends? p/Connectable block)))

;; # Block Hierarchy root

(def jdbc-datasource :jdbc/datasource)

;; # Migrations

(defmulti migrate-db!
  (fn [_ config]
    (:migration/type config)))

;; # configuration

(s/def ::jdbcUrl string?)
(s/def ::datasource (s/keys :req-un [::jdbcUrl]))
(s/def ::configuration (s/keys :req-un [::datasource]))

(def datasource-defaults
  {:datasource {:autoCommit false}})

;; # next.jdbc options

(defn ensure-options [ds {:keys [next-jdbc-options]}]
  (if next-jdbc-options
    (jdbc/with-options ds next-jdbc-options)
    ds))

;; # blocks helper

(defn start-hikaricp-pool [{:keys [db-spec creds] :as config}]
  (let [jdbc-url (when db-spec
                   {:jdbcUrl (njc/jdbc-url db-spec)})
        config   (update config :datasource
                   #(cb/deep-merge jdbc-url (cb/ensure-value creds) %))
        config   (cb/deep-merge
                   datasource-defaults
                   config)
        config   (cb/check ::configuration config)
        ds       (njc/->pool HikariDataSource (:datasource config))]
    (ensure-options ds config)))

(defn ensure-db-migrated! [ds config]
  (when-let [migration (:migration config)]
    (migrate-db! ds migration)))

(defn start-hikaricp-pool! [config]
  (let [ds (start-hikaricp-pool config)]
    (ensure-db-migrated! ds config)
    ds))

(defn halt! [ds]
  (let [ds (p/get-datasource ds)]
    (when (instance? Closeable ds)
      (.close ^Closeable ds))))

(defn data-source-config [^HikariDataSource ds]
  {:jdbcUrl              (.getJdbcUrl ds)
   :username             (.getUsername ds)
   :readOnly             (.isReadOnly ds)
   :driverClassName      (.getDriverClassName ds)
   :dataSourceProperties (into {} (.getDataSourceProperties ds))})

;; # main block

(derive ::datasource jdbc-datasource)

(defmethod cb/start-block!
  ::datasource
  [_ _ config _]
  (start-hikaricp-pool! config))

(defmethod cb/resolve-block
  ::datasource
  [_ ds]
  ds)

(defmethod cb/stop-block!
  ::datasource
  [_ ds]
  (halt! ds))
