# ceviche/bootstrap

A library of functions for generating Bootstrap-compatible `Hiccup` data.

# Installation

### Leiningen

Add to your `project.clj`:

```clojure
:dependencies [ ...
               [ceviche/bootstrap "0.1.0-SNAPSHOT"]]
```

## Usage

```clojure
(require '[bootstrap.core :as bs])

(bs/btn) ;=> ["button.btn.btn-default" {}]

(require '[hiccup.core :refer [html]])

(html (bs/btn)) ;=> "<button class="btn btn-default"></button>"
```

As you can see above, ceviche/bootstrap consists of functions (like `btn`) that fill in a lot of the CSS classes and other boilerplate that makes your `Hiccup` ready for Bootstrap.

All of the functions allow the following signatures (more detail below):

```clojure
(function-name css-string options-set attrs-map & content)
(function-name css-string options-set & content)
(function-name css-string attrs-map & content)
(function-name options-set attrs-map & content)
(function-name options-set & content)
(function-name attrs-map & content)
(function-name & content)
```

### css-string

```clojure
(bs/btn "#my-btn.btn-class") ;=> ["button#my-btn.btn-class.btn.btn-default" {}]
```

This is used to build a `Hiccup`-compatible CSS string.  **Do not include the tagname**.  You may provide an optional id and any number of classes.  [See here](https://github.com/weavejester/hiccup/wiki/Syntax#css-style-sugar) for formatting details.

Note that the CSS string is optional, but when it is included, it must be followed either by the `options-set` or the `attrs-map`.  This is because the first element of `& content` might be a string, so there has to be something between `css-string` and `& content` to make it clear that `css-string` is not a member of `& content`.

### options-set

```clojure
(bs/btn #{"input" "primary"}) ;=> ["input.btn.btn-primary" {}]

; combined with css-string for illustration:

(bs/btn "#my-btn" #{"input"}) ;=> ["input#my-btn.btn.btn-default" {}]
```

The `options-set` is a set containing options specific to each function.

Some functions take no options.  In that case, a set can still be supplied, since all functions accept the same signatures, but it will be ignored.

Most of the functions take certain predefined strings as options, as in the example above.  `"input"` and `"primary"` have special meanings when supplied to `btn`.

Some functions expect a map as one of the options:

```clojure
(bs/col #{{"md" [4] "lg" [3 4]}}) ;=> ["div.col-md-4.col-lg-3.col-lg-offset-4" {}]
```

Don't confuse the map that goes **inside** the `options-set` with `attrs-map`, which is also a map but goes **after** `options-set`.

Note that `options-set` is optional, but when it and `attrs-map` are both supplied, it goes before `attrs-map`.

### attrs-map

```clojure
(bs/btn {:data-property "value"})

;=> ["div.btn.btn-default" {:data-property "value"}]

(html (bs/btn {:data-property "value"}))

;=> "<button class="btn btn-default" data-property="value"></button>"
```

There is not much to say about `attrs-map`.  It is exactly the same as what you would supply to `Hiccup`, and in fact it is passed directly to `Hiccup`.  It is optional.

### & content

```clojure
(bs/col #{{"md" [4]}} "Hello, world!" [:p#my-p "My P element"])

;=> ["div.col-md-4" {} "Hello, world!" [:p#my-p "My P element"]]

(html (bs/col #{{"md" [4]}} "Hello, world!" [:p#my-p "My P element"]))

;=> "<div class="col-md-4">Hello, world!<p id="my-p">My P element</p></div>"
```

Like `attrs-map`, `& content` is passed directly to `Hiccup`.
