(ns kuro.fude
  (:use benri.kuro)
  (:require [clojure.string :as str]
            [clojure.core.match :refer [match]]))

(defonce training-data
  (str "後 に　就いて　言って　下さい。 もう　一度。 全部　一緒 に。 はじめ から。"
       "次 を　どうぞ。 よく　できる。 おはよう　ございます。 どうも　ありがとう　ございます。 "
       "どう　いたしまして。 お元気　です　か。 いかが　です　か。 元気　です。 まあまあ　です。 "
       "おかげさまで。 お蔭　様 で。 行って　来ます。 行って　いらっしゃい。 行ってらっしゃい。 "
       "ただいま。 お帰りなさい。 Ｌ　です。 まだまだ　暑い　日 が　続きますね。 "
       "でも　クーラー の　温度 の　下げすぎ は　体 に　よく　ありません。 "
       "今夜 は 「金曜特別　ロードショー」と　題して。 皆さん　お待ち　かね。 "
       "「デスノート　リライト　２　Ｌ を　継ぐ者」を　お送り　いたします。 続編　であり"
       "日本 に　行きます。 必ず　２７　度前後 に　して。 "
       "冷やしすぎない　よう に　して　ください。 皆さん　お待ち　かね。 死んだ　あと。 "))

(defn some-space
  "Returns space surfaceForm from token, or nil if token isn't any space."
  [token] (some #{(:surfaceForm token)} [" " "　"]))

(defn inject-space-info
  "Inserts space information (:prev-space/:next-space) into a token if the
   other token is a space."
  [prev-token next-token]
  [(or (some->> (some-space next-token) (assoc prev-token :next-space))
  	   prev-token)
   (or (some->> (some-space prev-token) (assoc next-token :prev-space))
   	   next-token)])

(defn- filter-by-some-part-of-speech
  [n {:keys [partOfSpeech]} tokens]
  (let [take-n (partial take n)
        split-speech #(str/split % #",")
        partOfSpeech (split-speech partOfSpeech)
        equal-speech #(= (take-n partOfSpeech) (-> % :partOfSpeech take-n))
        f (comp equal-speech #(update % :partOfSpeech split-speech))]
  	(seq (filter f tokens))))

(defn- filter-by-surface
  [{:keys [surfaceForm]} tokens]
  (seq (filter #(= (:surfaceForm %) surfaceForm) tokens)))

(defmulti elect-by (fn [& args] (first args)))

(defmethod elect-by :surface-and-part-of-speech
  [_ n token tokens]
  (let [take-n #(take n %)]
  	(->> (filter-by-some-part-of-speech n token tokens)
  		   (filter-by-surface token))))

(defmethod elect-by :part-of-speech [_ n token tokens]
  (filter-by-some-part-of-speech n token tokens))

(defmethod elect-by :surface [_ token tokens]
  (filter-by-surface token tokens))

(defn find-token-candidates
  [token the-model]
  "Accepts token and model returning a list of token candidates."
  (or (elect-by :surface-and-part-of-speech 4 token the-model)
      (elect-by :surface-and-part-of-speech 3 token the-model)
      (elect-by :surface-and-part-of-speech 2 token the-model)
      (elect-by :part-of-speech 4 token the-model)
      (elect-by :surface-and-part-of-speech 1 token the-model)
      (elect-by :part-of-speech 3 token the-model)
      (elect-by :part-of-speech 2 token the-model)
      (elect-by :surface token the-model)
      (elect-by :part-of-speech 1 token the-model)))

(defn remove-token-dups
  [tokens]
  (let [merge-dups (fn [prev nekt]
                     (if (= (:position prev) (:position nekt))
                       (merge prev nekt) [prev nekt]))]
    (loop [t (vec tokens) r []]
      (match [t r]
        [[] _] (reverse r)
        [[h1 & r1] []] (recur r1 [h1])
        [[h1 & r1] [h2 & r2]]
          (recur r1 (-> (merge-dups h1 h2) (cons r2) flatten vec))))))