(ns rightangle.mandrill.client
  (:require [clj-http.client :as client]
            [clojure.data.json :as json]
            [clojure.string :as string]))

(defrecord MandrillClient [clj-http-options])

(def ^:private api-key nil)
(def ^:private base-url "https://mandrillapp.com/api/1.0/")

(defn- build-endpoint
  [call]
  (str base-url call ".json"))

(defn- call
  [client endpoint body]
  {:pre [(instance? MandrillClient client)]}
  (let [body (assoc body :key (or (:key body api-key)))
        params (-> {:as :json
                    :content-type :json
                    :headers {"User-Agent" "rightangle-mandrill/0.1.0"}
                    :body (json/write-str body)}
                   (merge (:clj-http-options client)))]
    (:body (client/post (build-endpoint endpoint) params))))

(defn set-api-key!
  [key]
  (alter-var-root #'api-key (constantly key)))

(defn create
  [& [clj-http-options]]
  (let [clj-http-options (or clj-http-options {})]
    (->MandrillClient clj-http-options)))

(defn messages-send
  [client request]
  (call client "messages/send" request))

(defn messages-send-template
  [client request]
  (call client "messages/send-template" request))

(defn messages-search
  [client request]
  (call client "messages/search" request))

(defn messages-search-time-series
  [client request]
  (call client "messages/search-time-series" request))

(defn messages-info
  [client request]
  (call client "messages/info" request))

(defn messages-content
  [client request]
  (call client "messages/content" request))

(defn messages-parse
  [client request]
  (call client "messages/parse" request))

(defn messages-send-raw
  [client request]
  (call client "messages/send-raw" request))

(defn messages-list-scheduled
  [client request]
  (call client "messages/list-scheduled" request))

(defn messages-cancel-scheduled
  [client request]
  (call client "messages/cancel-scheduled" request))

(defn messages-reschedule
  [client request]
  (call client "messages/reschedule" request))
