(ns qbits.tape.cycle-listener
  (:refer-clojure :exclude [cycle])
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clojure.java.shell :as shell]
            [clojure.datafy :as d])
  (:import (net.openhft.chronicle.queue.impl StoreFileListener)
           (net.openhft.chronicle.queue RollCycle RollCycles)
           (java.io File FileOutputStream)
           (java.util.zip GZIPOutputStream)
           (java.time.format DateTimeFormatter DateTimeParseException)))

(defn gzip
  [^File file]
  (let [file-name (.getName file)
        parent (.getParent file)
        tar-file-name (format "%s.tar.gz" file-name)]
    (if (.exists (io/file (str parent "/" tar-file-name)))
      (log/infof "Skipping compression of %s" file-name)
      (do (log/infof "Compressing %s" file-name)
          ;; FIXME
          ;; shelling out like Calimero
          (shell/sh "sh" "-c" (format "tar -cvzf %s %s -C %s"
                                      tar-file-name
                                      file-name
                                      parent))))))

(defn file-formatter
  [fmt]
  (DateTimeFormatter/ofPattern (format "%s'.cq4'")))

(defn cycle-file?
  [^RollCycle roll-cycle file-name]
  (try
    (-> (format "%s'.cq4'" (.format roll-cycle))
        DateTimeFormatter/ofPattern
        (.parse file-name))
    (catch DateTimeParseException e
      nil)))

(defn delete
  [file]
  (log/infof "Deleting %s" file)
  (io/delete-file file))

(defn cycle-files
  "Returns seq of files sorted by date"
  [{:keys [dir roll-cycle]}]
  (let [files (file-seq (io/file dir))]
    (some->> files
             (filter #(->> ^File % .getName (cycle-file? roll-cycle)))
             sort
             reverse)))

(defmulti action (fn [act ctx] (:type act)))

(defmethod action :log
  [opt ctx]
  (run! #(log/info ::log (str %))
        (cycle-files ctx)))

(defmethod action :gzip
  [{:keys [keep-max]
    :or {keep-max 1}}
   ctx]
  (run! gzip
        (drop keep-max (cycle-files ctx))))

(defmethod action :delete
  [{:keys [keep-max]
    :or {keep-max 1}}
   ctx]
  (run! delete
        (drop keep-max (cycle-files ctx))))

(defn run-actions!
  [actions context]
  (run! #(action % context)
        actions))

(defn make
  ([dir roll-cycle]
   (make dir roll-cycle nil))
  ([dir roll-cycle actions]
   (let [ctx {:roll-cycle roll-cycle
              :dir dir}]
     (reify StoreFileListener
       (onReleased [_ cycle file]
         (log/infof "Got release on %s %s" cycle file)
         (run-actions! actions (assoc ctx
                                      :current-cycle cycle
                                      :file file))))
     (run-actions! actions ctx))))

;; (run-actions! [{:type :delete :keep-max 0}
;;                {:type :gzip :keep-max 0}
;;                {:type :log}
;;                ]
;;               {:dir "/tmp/q3" :roll-cycle RollCycles/DAILY})
