(ns qbits.tape.listener
  (:refer-clojure :exclude [cycle])
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clojure.java.shell :as shell]
            [clojure.datafy :as d])
  (:import (net.openhft.chronicle.queue.impl StoreFileListener)
           (net.openhft.chronicle.queue RollCycle)
           (java.io File FileOutputStream)
           (java.util.zip GZIPOutputStream)))

(defn gzip
  [^File file]
  (let [name (.getName file)
        parent (.getParent file)]
    (log/infof "Compressing %s" file)
    ;; FIXME
    ;; shelling out like Calimero
    (shell/sh "sh" "-c" (format "tar -cvzf %s.tar.gz %s -C %s" name name parent))))

(defn delete
  [file]
  (log/infof "Deleting %s" file)
  (io/delete-file file))

(defn match-cycle-file?
  [s]
  (re-find #"^\d{4}\d{2}\d{2}\.cq4$" s))

(defn cycle-files
  "Returns seq of files sorted by date"
  [queue]
  (some->> queue
           d/datafy
           :qbits.tape.queue/file
           io/file
           file-seq
           (filter #(-> ^File % .getName match-cycle-file?))
           sort
           reverse))

(defmulti action (fn [act & _] (:type act)))

(defmethod action :log
  [opt queue cycle file]
  (run! #(log/info ::log (str %))
        (cycle-files queue)))

(defmethod action :gzip
  [{:keys [keep-max]
    :or {keep-max 1}}
   queue cycle file]
  (run! gzip
        (drop keep-max (cycle-files queue))))

(defmethod action :delete
  [{:keys [keep-max]
    :or {keep-max 1}} queue cycle file]
  (run! delete
        (drop keep-max (cycle-files queue))))

(defn run-actions!
  [actions queue cycle file]
  (run! #(action % queue cycle file)
        actions))

(defn make
  ([queue]
   (make queue nil))
  ([queue actions]
   (reify StoreFileListener
     (onReleased [_ cycle file]
       (log/infof "Got release on %s %s" cycle file)
       (run-actions! actions queue cycle file )))
   (run-actions! actions queue nil nil)))

;; (run-actions! [{:type :delete :keep-max 5}
;;                {:type :gzip :keep-max 2}
;;                {:type :log}]
;;               qbits.tape/q nil nil)
