(ns geppetto.repeat
  (:use [clojure.pprint :only [pprint]])
  (:use [clojure.data :only [diff]])
  (:use [geppetto.runs])
  (:use [geppetto.records])
  (:use [geppetto.parameters]))

(defn repeat-run
  "Returns results. Probably should be used by (verify-identical-repeat-run)."
  [runid run-fn datadir git recordsdir nthreads]
  (let [run (get-run runid)
        params-string (format "{:control %s :comparison %s}"
                         (prn-str (:control run))
                         (prn-str (:comparison run)))
        params (read-params params-string)]
    (println (format "Repeating run %d with parameters:" runid))
    (pprint params)
    (run-with-new-record run-fn params-string
      datadir (:seed run) git recordsdir nthreads
      (:repetitions run) false true true)))

(defn extract-single
  [rs resultstype only-ignore]
  (let [{:keys [only ignore]} (get only-ignore resultstype)]
    (map (fn [r] (cond only (select-keys r only)
                    ignore (apply dissoc r ignore)
                    :else r))
       rs)))

(defn extract-relevant-results
  [results only-ignore]
  (for [{:keys [control comparison comparative]} results]
    {:control (extract-single control :control only-ignore)
     :comparison (extract-single comparison :comparison only-ignore)
     :comparative (extract-single comparative :comparative only-ignore)}))

(defn verify-identical-repeat-run
  "only-ignore parameter takes the format:
   {:control {:only [:key1 :key2]} :comparison {:ignore [:key1 :key2]}} etc.
   Only takes priority over ignore."
  [runid only-ignore run-fn datadir git nthreads]
  (let [old-results (doall
                     (extract-relevant-results
                      (sort-by (comp :simid :control)
                               (get-raw-results runid))
                      only-ignore))
        _ (println "old-results:" old-results)
        new-results (doall
                     (extract-relevant-results
                      (sort-by (comp :simid :control)
                               (repeat-run runid run-fn datadir git "/tmp" nthreads))
                      only-ignore))
        _ (println "new-results: new-results")]
    (filter identity
       (for [[old-sim new-sim] (partition 2 (interleave old-results new-results))]
         (let [[only-in-old only-in-new _] (diff old-sim new-sim)]
           (when (or only-in-old only-in-new)
             {:old old-sim :new new-sim :diffs [only-in-old only-in-new]}))))))
