(ns {{name}}.core (:gen-class)
  (:require [cascalog.tap :refer [hfs-tap lfs-tap]]
            [cascalog.api :refer [<- ?- stdout select-fields hfs-textline lfs-textline with-job-conf compile-flow combine]]
            [cascalog.ops :refer [sum]]
            [cascalog.more-taps :refer [lfs-delimited hfs-delimited]]
            [clojure.tools.cli :refer [cli]])
  (:import [cascading.avro AvroScheme]))

(defn parse-int [string]
  (Integer/parseInt string))

(defn query [tsv-input-tap avro-input-tap trap-tap]
  (let [number-1-source (<- [?number]
          (tsv-input-tap :> _ ?number-string)
          (parse-int :< ?number-string :> ?number))
        number-2-source (<- [?number]
          (avro-input-tap :> _ ?number))]
    (<- [?sum]
        (:trap trap-tap)

        ((combine number-1-source number-2-source) :> ?number)
        (sum :< ?number :> ?sum))))

(def remote-query
    (query
      (hfs-delimited "hdfs:///user/luser/inputs/tsv/")
      (let [avro-tap (hfs-tap (AvroScheme.) "hdfs:///user/luser/inputs/avro/")]
        (<- [?letter ?number]
            ((select-fields avro-tap ["letter" "number"]) :> ?letter ?number)))
      (hfs-textline "hdfs:///user/luser/TRAPFILE.trap")))

(def local-query
    (query
      (lfs-delimited "resources/local/input.tsv")
      (let [avro-tap (lfs-tap (AvroScheme.) "resources/local/input.avro")]
        (<- [?letter ?number]
            ((select-fields avro-tap ["letter" "number"]) :> ?letter ?number)))
      (lfs-textline "TRAPFILE.trap")))

(defn -main [& cmd-args]
  (let [[options args banner]
        (cli cmd-args
             ["-h" "--help" "Show help" :default false :flag true]
             ["-d" "--diagram" "Draw flow diagram." :default false :flag true]
             ["-l" "--local" "Use local data." :default false :flag true]
             ["-h" "--remote" "Use remote data." :default false :flag true])]
    (when (:help options)
      (println banner)
      (System/exit 0))

    (when (not (or (:local options) (:remote options) (:diagram options)))
      (println "Either local, remote, or diagram must be specified.")
      (System/exit 1))

    (cond
       (:remote options)
         (with-job-conf {"mapred.reduce.tasks" 16}
           (?- "JOB NAME"
               (hfs-textline "hdfs:///user/luser/OUTPUT.tsv")
                remote-query))
       (:local options)
         (?- "JOB NAME"
             (lfs-textline "OUTPUT.tsv")
             local-query)
       (:diagram options)
         (.writeDOT
           (compile-flow
             (lfs-textline "OUTPUT.tsv")
              local-query)
              "flow.dot"))))
