(ns caribou.plugin.rss
  (:require [caribou.model :as model]
            [caribou.plugin.protocol :as plugin]
            [clojure.java.io :as io]
            [clojure.edn :as edn]
            [clj-rss.core :as rss]
            [clojure.xml :as xml])
  (:import (java.io PushbackReader)))

(defrecord FeedPlugin [location route get-stories title description])

(defn channel-data
  [feed-plugin]
  (-> (select-keys feed-plugin [:title :description])
      (assoc :link (str (:location feed-plugin) \/ (:route feed-plugin)))))

(defn emit
  "emit xml from edn"
  [data]
  (with-out-str (xml/emit data)))

(defn migrate
  ;; todo : make this do something?
  [& args])

(def dummy-stories
  (-> "caribou/plugin/rss/dummy-input.edn"
      io/resource
      io/reader
      PushbackReader.
      edn/read))

(defn make-feed-item
  [feed-plugin story]
  (let [extracted (select-keys story [:title :link :description])
        massaged (assoc extracted
                   :pubDate (:created-at story)
                   :author (str (:email story) " (" (:author story) \))
                   :guid (str (:location feed-plugin) \/ (:uuid story)))]
    massaged))

(defn get-stories
  "Replace this with your own feed extracting function, the output should
   have the following format:
   {:pubDate date of creation
    :author \"someone@example.com (real name)\"
    :guid URL string that globally, uniquely identifies the story
    :title title of the document
    :link canonical link to the story
    :description a short description of the story}"
  [this config]
  (map (partial make-feed-item this) dummy-stories))

(defn feed-core
  "create an rss feed"
  [feed-plugin config]
  (fn [request]
    (let [stories ((:get-stories feed-plugin) feed-plugin config)
          feed (rss/channel (channel-data feed-plugin) stories)]
      {:headers {"Content-Type" "application/rss+xml"}
       :status 200
       :body (emit feed)})))

(def default-impl
  {:location "http://localhost:33333"
   :route "feed.rss"
   :get-stories get-stories
   :title "Caribou Development"
   :description "The Caribou Development Blog"})

(defn create
  [implementation config]
  (map->FeedPlugin (merge default-impl implementation)))

(plugin/make
 FeedPlugin
 {:migrate migrate
  :provide-pages (fn [this config]
                   {:rss [{:path (str \/ (:route this))
                           :name "Rss Feed"
                           :slug "rss-feed"
                           :template ""
                           :controller 'caribou.plugin.rss
                           :action (feed-core this config)}]})})
