(ns ring.middleware.honeybadger
  (:require [honeybadger.core :as hb]
            [ring.util.request :as req]))

(defn- request-params
  "Determine params map to send to Honeybadger. Uses value of :params
  if present, else merges :query-params and :form-params.

  Rationale is: (1) w/o ring.middleware.params, :params may not exist
  at all, and (2) always merging :query-params and :form-params is not
  reliable as (e.g.) JSON params may only be merged into :params (a la
  ring.middleware.json)."
  [request]
  (if (contains? request :params)
    (:params request)
    (merge (:query-params request) (:form-params request))))

(defn- request->metadata
  "Given a Ring request, extract and format the key details as
  honeybadger metadata."
  [request]
  {:request {:method  (:request-method request)
             :url     (req/request-url request)
             :params  (request-params request)
             :session (:session request)}})

(defn notify
  "Notify Honeybadger of error.  Exists to make testing
  easier (i.e. using `with-redefs`)."
  [config ex metadata]
  @(hb/notify config ex metadata))

(defn- bounced-error? [ex]
  (::bounced-request (ex-data ex)))

(defn- bounce [ex req]
  (ex-info "Bouncing error" {::bounced-request req} ex))

(defn wrap-honeybadger
  "Ring middleware to report handler exceptions to
  honeybadger.io. :api-key is the only required option."
  [handler options]
  (fn [request]
    (try
      (handler request)
      (catch Throwable t
        (let [{:keys [callback context-fn]} options
              [t request] (if (bounced-error? t)
                            [(.getCause t) (::bounced-request (ex-data t))]
                            [t request])
              hb-options (select-keys options [:api-key :env :filters])
              metadata (cond-> (request->metadata request)
                         context-fn (assoc :context (context-fn request)))
              hb-id (notify hb-options t metadata)]
          (when callback
            (callback t hb-id)))
        (throw t)))))

(defn wrap-honeybadger-bouncer
  "Ring middleware which can be installed later in the middleware
  chain to allow taking advantage of later processing of the request
  by middleware.  Will \"bounce\" the more fully processed request up
  to stack to the reporter, via a specialized exception type.  Can be
  wrapped multiple times will no ill effects."
  [handler]
  (fn [request]
    (try
      (handler request)
      (catch Throwable t
        (throw (if (bounced-error? t) t (bounce t request)))))))
