(ns ca.bigthunder.libs.logging
  "Helpers wrapping timbre logging library."
  {:author "twocats@bigthunder.ca"}
  (:require
   [taoensso.timbre :as timbre :refer [*config*]]))

(def default-timestamp-opts timbre/default-timestamp-opts)
(def default-output-fn      timbre/default-output-fn)

(def defaults
  {:level          :debug
   :ns-whitelist   []
   :ns-blacklist   []
   :timestamp-opts default-timestamp-opts
   :output-fn      default-output-fn
   :appenders
    {:println (timbre/println-appender {:stream :auto})}})

(timbre/set-config! defaults)

(defmulti regime (fn [k & ks] (identity k)))

(let [global timbre/set-level!
      only   (fn [ks k]
              (run! #(timbre/merge-config! {:appenders {% {:min-level k}}}) ks)
              *config*)]
 (defmethod regime :paranoid
   ([_]    (global :trace))
   ([_ ks] (only ks :trace)))
 (defmethod regime :development
   ([_]    (global :debug))
   ([_ ks] (only ks :debug)))
 (defmethod regime :production
   ([_]    (global :info))
   ([_ ks] (only ks :info)))
 (defmethod regime :quiet
   ([_]    (global :error))
   ([_ ks] (only ks :error)))
 (defmethod regime :super-quiet
   ([_]    (global :fatal))
   ([_ ks] (only ks :fatal))))

(defmulti add (fn [k s] (identity k)))

(letfn [(push [k v]
         (timbre/merge-config! {k (conj (*config* k) v)}))]

 (defmethod add :to.blacklist [_ s]
   (push :ns-blacklist s))

 (defmethod add :to.whitelist [_ s]
   (push :ns-whitelist s)))

(defmethod add :to.appenders [_ m]
  (timbre/merge-config! {:appenders {(m :key) (m :fn)}}))

(defn with-file [fname]
  (add :to.appenders
    {:key :spit
     :fn (timbre/spit-appender {:fname fname})}))

(defn reset []
 (timbre/set-config! defaults))

(defn appenders [] (keys (*config* :appenders)))

(letfn
 [(switcher [b] #(timbre/merge-config! {:appenders {% {:enabled? b}}}))
  (set-running ([b]
                (let [f (switcher b)]
                 (run! f (appenders)))
                *config*)
               ([b ks]
                (let [f (switcher b)]
                 (run! f ks))
                *config*))]

 (defn off ([]   (set-running false))
           ([ks] (set-running false ks)))
 (defn on  ([]   (set-running true))
           ([ks] (set-running true ks))))
