(ns retabled.core
  (:require [retabled.filter :as filter]
            [retabled.helpers :as helpers]
            [retabled.listeners :as listeners]
	    [retabled.sort :as sort]
            #?(:cljs [reagent.core :refer [atom]])
            [clojure.string :as str]))

(def col-map-help
  "Possible values of col-maps within `:columns` of control-map "
  [{:valfn (fn [entry] "Retrieves the value of this cell from `entry`")
    :sortfn (fn [entry] "Given an entry, how to sort it by this field. Defaults to `valfn`.")
    :displayfn (fn [valfn-value] "Produces the display from result `(valfn entry)`. 
                                  Default `identity`")
    :headline "The string to display in table header"
    :css-class-fn (fn [entry] "Produces class (str or vector) to be applied to field/column")
    :ignore-case? "Whether to ignore the case during filtering"
    :filter "If truthy displays a filter-bar that will perform on-change filtering on this column.
             If set to :click-to-filter, makes values searchable by clicking them"
    :filter-in-url "If false turns off filtering in url for the column, by default true"}])

(def control-map-help
  "The possible values of a control-map for a table, which should be a
  sequence of maps where each map corresponds to one column of the table."
  {:row-class-fn (fn [entry] "Provides the class (str or vector) of a :tr, given entry")
   :columns col-map-help
   :controls-left (fn [content] "fn of `content` to place before the paging controls (if any)")
   :controls-right (fn [content] "fn of `content` to place after the paging controls (if any)")
   :default-styling? "If truthy, apply default styles such as direction indicators on sorts"
   :table-id "Allows for each table to have a unique ID"
   :filter-in-url "If false turns off filtering in url for the table, by default true"
   :paging {:simple "If truthy, use a local atom and default setters and getters without bothering with anything else defined in `:paging`. "
            :get-current-screen (fn [] "Get the current screen num (0-based), from an atom or reframe, etc")
            :set-current-screen (fn [n] "Set current screen num. Default 0.")
            :get-final-screen (fn [] "Get the final screen num (0-based), from an atom or reframe, etc")
            :set-final-screen (fn [n] "Set final screen num.")
            :get-amount (fn [] "Get the number of entries visible per screen")
            :set-amount (fn [n] "Set number of entries visible per screen. Default 5.")
            :r-content [:div.icon "prev-page"]
            :rr-content [:div.icon "first page"]
            :f-content [:div.icon "next page"]
            :ff-content [:div.icon "final page"]
            :show-total-pages? "If truthy, it will show the current page and total pages in the middle content as opposed to just the current page."
            :left-bar-content [:div.whatever "Stuff before the controls"]
            :right-bar-content [:div.whatever "Stuff after the controls"]
            :entries-option {:values ["A vector of integers representing options for how many entries appear per page"]
                             :default "An integer showing the default value for how many entries appear per page"}
            :align "Choose either `left` or `right` or `center` for paging controls alignment. Default to `left`"}
   :table-scroll-bar {:fixed-columns {:first? "If truthy, the first column of the table will be fixed/sticky"
                                      :last? "If truthy, the last column of the table will be fixed/sticky"
                                      :table-color "Changes default values for the background colors to match different colored tables"}}
   :example/csv-download? "If truthy, a button will appear with the corresponding table for downloading the data in a csv file"
   :sort-filter-highlight "If truthy, columns will be highlighted when they are currently sorted or filtered. The default highlight color is light gray, but a string can be placed here as the truthy value to customize that color."})

(def PAGING-INFO (atom {:entries-per-page 0
                        :go-to-value "1"
                        :current-page 1}))

(def ALL-DATA (atom {}))

(def CURRENT-DATA (atom {}))

(defn atom?
  "ducktype an atom as something dereferable"
  [a]
  (try (do (deref a) true)
       (catch #?(:clj Exception :cljs js/Error) _ false)))

(defn ^{:private true} render-header-fields
  [{:as args
    :keys [controls
           SORT
           FILTER
           table-id]}]
  (into [:tr.table-headers.row]
        (for [c (:columns controls)
              :let [h  (cond->> (:headline c)
                         (:sort c) (sort/gen-sort c SORT))
                    fi (when (:filter c) (filter/gen-filter {:col-map c
                                                             :FILTER  FILTER
                                                             :table-id  table-id
                                                             :filter-in-url (:filter-in-url controls)}))
                    table-color (get-in controls [:table-scroll-bar :table-color] "white")
                    style-first-column-fixed {:style {"position" "sticky"
                                                        "left" "0"
                                                        "backgroundColor" table-color}}
                      style-last-column-fixed {:style {"position" "sticky"
                                                       "right" "0"
                                                       "backgroundColor" table-color}}
                      first-column-and-first-is-fixed? (fn [column] (and (= column (first (:columns controls))) (get-in controls [:table-scroll-bar :first?])))
                    last-column-and-last-is-fixed? (fn [column] (and (= column (last (:columns controls))) (get-in controls [:table-scroll-bar :last?])))
                    sort-filter-highlight (:sort-filter-highlight controls)
                      currently-sorted-or-filtered? (fn [column] (or (and (:selected @SORT)(= (:sortfn column) (:selected @SORT)))
                                                                     (= (:valfn column) (:selected @SORT))
                                                                     (> (count (get-in @FILTER [(:valfn column) :value])) 0)))
                      style-th (assoc-in (if (first-column-and-first-is-fixed? c)
                                           style-first-column-fixed
                                           (if (last-column-and-last-is-fixed? c)
                                             style-last-column-fixed))
                                         [:style "backgroundColor"]
                                         (if (and sort-filter-highlight (currently-sorted-or-filtered? c))
                                           (if (string? sort-filter-highlight)
                                             sort-filter-highlight
                                             "rgb(240, 240, 240)")
                                           table-color))]]
        [:th style-th fi h])))

(defn ^{:private true} render-screen-controls
  "Render the controls to edit this screen for results"
  [{:as paging-controls
    :keys [get-current-screen
           get-amount
           set-amount
           set-current-screen
           set-final-screen
           get-final-screen
           r-content
           rr-content
           f-content
           ff-content
           left-bar-content
           right-bar-content
           num-columns
           entries-option
           align
           show-total-pages?]
    :or {num-columns 100}}
   table-scroll-bar]
  (let [current-screen-for-display (inc (get-current-screen))
        prevfn #(max (dec (get-current-screen)) 0)
        nextfn #(min (inc (get-current-screen)) (get-final-screen))
        first-page? (= (get-current-screen) 0)
        last-page? (= (get-current-screen) (get-final-screen))
        style-hidden {:style {"color" "transparent"}
                      :class "hidden"}
        style-page-to-go {:style {"width" "3em"
                                  "marginLeft" ".5em"}
                          :value (:go-to-value @PAGING-INFO)
                          :on-click #(swap! PAGING-INFO assoc :go-to-value "")
                          :id "page-to-go"}
        on-change-page-to-go (fn [evt]
                               (let [val (-> evt .-target .-value)
                                     int-val (int val)]
                                 (when (= val "")
                                   (swap! PAGING-INFO assoc :go-to-value val))
                                 (when (and (> int-val 0)(<= int-val (+ (get-final-screen) 1)))
                                   (do (swap! PAGING-INFO assoc :go-to-value (str int-val))
                                       (set-current-screen (- int-val 1))))))
        table-color (:table-color table-scroll-bar "white")]
    [:tr.row.screen-controls-row
     [:td.cell.screen-controls {:colSpan num-columns :style {"borderColor" table-color "textAlign" align}}
      [:div.control (when table-scroll-bar {:style {"position" "sticky" align "1em"}})
       left-bar-content
       [:div.control.first [:a.control-label (if first-page?
                                              style-hidden
                                              {:on-click #(set-current-screen 0)})
                           rr-content]]
      [:div.control.prev [:a.control-label (if first-page?
                                             style-hidden
                                             {:on-click #(set-current-screen (prevfn))})
                          r-content]]
       [:div.control.current-screen [:span.screen-num (if show-total-pages?
                                                        (str current-screen-for-display " of " (+ (get-final-screen) 1))
                                                        current-screen-for-display)]]
      [:div.control.next [:a.control-label (if last-page?
                                             style-hidden
                                             {:on-click #(set-current-screen (nextfn))})
                          f-content]]
      [:div.control.final [:a.control-label (if last-page?
                                              style-hidden
                                              {:on-click #(set-current-screen (get-final-screen))})
                           ff-content]]
      [:span.go-to "Go to"]
       [:input.page-to-go (assoc style-page-to-go :on-change on-change-page-to-go)]]
      (when (and entries-option (:values entries-option))
        (let [default (if (:default entries-option)
                        (:default entries-option)
                        (first (:values entries-option)))]
          (into [:select.entries-per-page {:defaultValue default
                                           :onChange (fn [evt]
                                                       (let [val (int (-> evt .-target .-value))]
                                                         (swap! PAGING-INFO assoc :entries-per-page val)
                                                         (set-current-screen 0)
                                                         (swap! PAGING-INFO assoc :go-to-value "1")))}]
                (for [num (:values entries-option)
                      :let [val-map {:value num}]]
                  [:option val-map  num]))))
      right-bar-content]]))

(defn generate-theads
  "generate the table headers"
  [{:as args
    :keys [controls
           paging-controls
           SORT FILTER
           table-id]}]
  (let [table-color (get-in controls [:table-scroll-bar :table-color] "white")]
    [:thead (when (:table-scroll-bar controls)
              {:style {"position" "sticky"
                       "top" "0"
                       "backgroundColor" table-color
                       "zIndex" "1"}})
     (when (:example/csv-download? controls)
     (helpers/render-csv-button-row @CURRENT-DATA table-id))
     (when (:paging controls)
       (render-screen-controls paging-controls (:table-scroll-bar controls)))
     (render-header-fields {:controls controls
                            :SORT  SORT
                            :FILTER  FILTER
                            :table-id  table-id})]))

(defn generate-rows
  "Generate all the rows of the table from `entries`, according to `controls`"
  [{:as args
    :keys [controls
           entries SORT
           FILTER
           table-id]}]
  (let [{:keys [row-class-fn columns]
         :or {row-class-fn (constantly "row")}} controls
        table-color (get-in controls [:table-scroll-bar :table-color] "white")]
    (into [:tbody]
          (for [e entries :let [tr ^{:key (gensym e)} [:tr {:class (row-class-fn e)}]]]
            (into tr
                  (for [c columns :let [{:keys [valfn css-class-fn displayfn filter]                                         
                                         :or {css-class-fn (constantly "field")
                                              displayfn identity}} c
                                        arg-map (cond-> {:class (css-class-fn e)}
                                                  (= filter :click-to-filter) (assoc :on-click (filter/on-click-filter {:col-map c
                                                                                                                        :table-id  table-id
                                                                                                                        :filter-in-url (:filter-in-url controls)
                                                                                                                        :FILTER  FILTER :value  (filter/resolve-filter c e)}))
                                                  (= filter :click-to-filter) (assoc :class (str (css-class-fn e) " click-to-filter")))
                                                  style-first-column-fixed {"position" "sticky"
                                                       "left" "0"
                                                                  "backgroundColor" table-color}
                                        style-last-column-fixed {"position" "sticky"
                                                         "right" "0"
                                                                 "backgroundColor" table-color}
                                        first-column-and-first-is-fixed? (and (= c (first columns)) (get-in controls [:table-scroll-bar :first?]))
                                        last-column-and-last-is-fixed? (and (= c (last columns)) (get-in controls [:table-scroll-bar :last?]))
                                        currently-sorted-or-filtered? (or (and (:selected @SORT)(= (:sortfn c) (:selected @SORT)))
                                                (= valfn (:selected @SORT))
                                                (> (count (get-in @FILTER [valfn :value])) 0))
                                        sort-filter-highlight (:sort-filter-highlight controls)
                                        cell-style-and-arg-map (assoc-in (if first-column-and-first-is-fixed?
                                (assoc arg-map :style style-first-column-fixed)
                                (if last-column-and-last-is-fixed?
                                  (assoc arg-map :style style-last-column-fixed)
                                  arg-map))
                              [:style "backgroundColor"]
                              (if (and sort-filter-highlight currently-sorted-or-filtered?)
                                (if (string? sort-filter-highlight)
                                  sort-filter-highlight
                                  "rgb(240, 240, 240)")
                                table-color))]]
                    ^{:key c} [:td.cell cell-style-and-arg-map
                               (-> e valfn displayfn)]))))))

(def DEFAULT-PAGE-ATOM (atom {:current-screen 0
                              :final-screen 0
                              :per-screen 10}))

(defn default-paging
  "Set up a local atom and define paging functions with reference to it"
  []
  (let [paging {:get-current-screen #(:current-screen @DEFAULT-PAGE-ATOM)
                :set-current-screen #(do (swap! DEFAULT-PAGE-ATOM assoc :current-screen %)
                                         (swap! PAGING-INFO assoc :current-page (+ 1 %))
                                         (swap! PAGING-INFO assoc :go-to-value (str (+ 1 %))))
                :get-amount #(if (> (:entries-per-page @PAGING-INFO) 0)
                               (:entries-per-page @PAGING-INFO)
                               (:per-screen @DEFAULT-PAGE-ATOM))
                :set-amount #(swap! DEFAULT-PAGE-ATOM assoc :per-screen %)
                :get-final-screen #(:final-screen @DEFAULT-PAGE-ATOM)
                :set-final-screen #(swap! DEFAULT-PAGE-ATOM assoc :final-screen %)
                :r-content "‹"
                :rr-content "«"
                :f-content "›"
                :ff-content "»"
                :align "left"}]
    paging))

(defn ^{:private true} paging
  "Limit view of entries to a given screen.
  If `paging-controls` is falsy, do not filter."
  [paging-controls entries]
  (if-not paging-controls entries
          (let [{:keys [get-current-screen
                        get-amount
                        set-amount
                        set-current-screen
                        set-final-screen
                        get-final-screen]} paging-controls
                amt (get-amount)
                parted-entries (if (> amt (count entries))
                                 (list entries)
                                 (partition amt amt nil entries))
                max-screens (dec (count parted-entries))]                  
              (set-final-screen max-screens)
              (nth parted-entries (get-current-screen)))))

(defn curate-entries
  [{:as args
    :keys [paging-controls entries SORT FILTER]}]
  (when (not-empty entries)
    (->> entries
         (paging paging-controls)
         (filter/filtering FILTER)
         (sort/sorting SORT))))

(def TABLE-NAME-COUNT (clojure.core/atom 0))

(defn check-for-duplicates
  [table-id]
  (if (< 1 #?(:clj 0 :cljs (count (js/Array.from (js/document.querySelectorAll (str "#" table-id))))))
    (throw #?(:clj (Exception. "multiple tables with the same table-id") :cljs (js/Error. "multiple tables with the same table-id")))
    ()))

(defn generate-table-id
  [table-id]
  (if-not table-id
    (do
      (swap! TABLE-NAME-COUNT inc)
      (str "__retabled-" @TABLE-NAME-COUNT))
    (do
      (try
        (check-for-duplicates table-id)
        (catch #?(:clj Exception :cljs js/Error) err
          #?(:clj (println err) :cljs (js/console.error err))))
      table-id)))

(defn update-all-data!
  [table-id entries]
  (swap! ALL-DATA assoc table-id entries))

(defn update-current-data!
  [table-id entries]
  (swap! CURRENT-DATA assoc table-id entries))

(defn update-default-entries-per-page!
  [controls]
  (let [entries-option (get-in controls [:paging :entries-option])
        default (get entries-option :default)
        values (get entries-option :values)
        num (if default
              default
              (first values))]
    (when (and entries-option (or default values))
      (swap! PAGING-INFO assoc :entries-per-page num))))

(defn table
  "Generate a table from `entries` according to headers and getter-fns in `controls`"
  [controls entries]
  (when (:paging controls)
    (listeners/activate PAGING-INFO))
  (let [SORT (atom sort/default-sort)
        FILTER (atom {})
        table-id (generate-table-id (:table-id controls))]
    (update-all-data! table-id entries)
    (update-default-entries-per-page! controls)
    (fn interior-table [controls entries]
      (let [paging-controls (cond (get-in controls [:paging :simple])
                                  (default-paging)

                                  (get-in controls [:paging])
                                  (merge (default-paging)
                                         (:paging controls))

                                  :no-paging
                                  nil)
            style-table {"height" "28em"
                         "width" "fit-content"
                         "maxWidth" "100%"
                         "display" "block"
                         "overflowY" "scroll"
                         "overflowX" "scroll"
                         "marginBottom" "3em"
                         "borderCollapse" "separate"}
            entries (curate-entries {:paging-controls paging-controls
                                     :entries  entries
                                     :SORT  SORT
                                     :FILTER  FILTER})]
        (update-current-data! table-id entries)
        [:table.table (assoc {:id table-id} :style (when (:table-scroll-bar controls)
                        style-table))
         [generate-theads {:controls controls
                           :paging-controls  paging-controls
                           :SORT  SORT
                           :FILTER  FILTER
                           :table-id  table-id}]
         [generate-rows {:controls controls
                         :entries  entries
                         :SORT SORT
                         :FILTER FILTER
                         :table-id table-id}]]))))
