(ns {{ns-name}}.layout
  (:require
   [hiccup.core :as hc]
   [hiccup.page :as hp]
   [hiccup.util :refer [with-base-url]]
   [hiccup.element :refer [javascript-tag]]
   [ring.util.http-response :refer [content-type ok]]
   [ring.util.anti-forgery :refer [anti-forgery-field]]
   [ring.middleware.anti-forgery :refer [*anti-forgery-token*]]))

(declare ^:dynamic *app-context*)

(def style-path "/css/")
(def script-path "/js/")
(def assets-path "/assets/")

(defn anti-forgery-element []
  [:input {:id "token" :value *anti-forgery-token* :type "hidden"}
   (javascript-tag (str  "var csrfToken = '" *anti-forgery-token* "'"))])

(defn boiler-header [& [username]]
  [:head [:title "ODH New Project"]
   (javascript-tag (str "var USERNAME = '" username "'"))
   (javascript-tag (str "var context = '" *app-context* "'"))])

(defn cljs-app-base []
  [:div#app
   [:div.container]])

(defn cljs-app-footer []
  [:div#footer
   [:div.footer]])

(defn boiler-plate [& [username]]
  [:div#boiler-wrapper
   [:div#navbar]
   ;; styles
   (hp/include-css (str assets-path "bootstrap/css/bootstrap.min.css"))
   (hp/include-css (str assets-path "font-awesome/css/font-awesome.min.css"))
   (hp/include-css (str style-path "style.css"))])

(defn cljs-includes []
  ;; script
  [:div
   (hp/include-js (str script-path "app.js")) ;; must precede the goog.require
   (hp/include-js (str assets-path "jquery/jquery.min.js"))
   (hp/include-js (str assets-path "tether/dist/js/tether.min.js"))
   (hp/include-js (str assets-path "bootstrap/js/bootstrap.min.js"))
   [:script {:type "text/javascript"} "goog.require('humfunding.app')"]])

(defn hiccup-render-cljs-base
  "Hiccup rendering (no traditional template)"
  [& [username]]
  (content-type
   (ok
    (hp/html5
     (boiler-header username)
     (boiler-plate username)
     (anti-forgery-element)
     (cljs-app-base)
     (cljs-app-footer)
     (cljs-includes)))
   "text/html; charset=utf-8"))

(defn error-page
  "error-details should be a map containing the following keys:
   :status - error status
   :title - error title (optional)
   :message - detailed error message (optional)

   returns a response map with the error page as the body
   and the status specified by the status key"
  [error-details]
  {:status  (:status error-details)
   :headers {"Content-Type" "text/html; charset=utf-8"}
   :body (hp/html5
          (boiler-header)
          (boiler-plate)
          [:div.alert.alert-warning
           [:h1 (or (:title error-details) (str "Error " (:status error-details)))]
           [:div.error-details (:message error-details)]])})
