(ns burningswell.worker.weather.models
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather.models :as models]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.producer :as producer]
            [burningswell.worker.topics :as topics]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [commandline.core :as cli]
            [datumbazo.core :as sql]
            [environ.core :refer [env]]
            [jackdaw.streams :as j]
            [netcdf.dods :as dods]
            [no.en.core :refer [split-by-comma]]
            [peripheral.core :as p]
            [taoensso.timbre :as log]))

(defn config
  "Returns the configuration for the app."
  [& [opts]]
  (->> {:application
        {"application.id" "update-weather-model"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"}
        :input {:commands topics/weather-model-update}
        :output {:succeeded topics/weather-model-update-succeeded}}
       (merge opts)))

(defn- latest-reference-time
  [datasources]
  (last (sort (map :reference-time datasources))))

(defn update-latest-reference-time!
  "Update the latest reference time of `model`."
  [db {:keys [datasources] :as model}]
  (let [reference-time (latest-reference-time datasources)]
    (->> (first @(sql/update db :weather.models
                   {:latest-reference-time reference-time}
                   (sql/where `(= :id ~(:id model)))
                   (sql/returning :*)))
         (merge model))))

(defn- datasource->row
  "Return a data source row for `model` and `datasource`."
  [model datasource]
  (-> (select-keys datasource [:das :dds :dods :reference-time])
      (assoc :model-id (:id model))))

(defn insert-datasources!
  "Update the data sources of `model`."
  [db  {:keys [datasources] :as model}]
  (->> @(sql/insert db :weather.datasources []
          (sql/values (map #(datasource->row model %) datasources))
          (sql/on-conflict [:model-id :reference-time]
            (sql/do-nothing))
          (sql/returning :*))
       (assoc model :datasources)))

(s/fdef insert-datasources!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

(defn update-model!
  "Update the data sources and the latest reference time of `model`."
  [db model]
  (log/info {:msg "Updating weather model."
             :model (select-keys model [:id :name :dods])})
  (let [model (merge model (models/by-id db (:id model))) ;; TODO: Remove after https switch
        {:keys [datasources] :as model}
        (->> (assoc model :datasources (dods/datasources model))
             (update-latest-reference-time! db)
             (insert-datasources! db))]
    (log/info {:msg "Updated weather model."
               :model (select-keys model [:id :name])
               :datasources (count datasources)})
    model))

(s/fdef update-model!
  :args (s/cat :db sql/db? :model :burningswell.db.weather.models/model))

;; Kafka Streams

(defn build-topology
  "Build the Kafka Streams topology."
  [{:keys [db config] :as app} builder]
  (-> (j/kstream builder (-> config :input :commands))
      (j/map-values #(update-model! db %))
      (j/to (-> config :output :succeeded))))

(defrecord Worker [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn worker
  "Returns a new weather model worker."
  [& [opts]]
  (-> (map->Worker {:config (config opts)})
      (component/using [:db :driver])))

;; Command line client

(defn client
  "Returns a client for the weather model worker."
  [{:keys [db producer] :as config}]
  (component/system-map
   :db (db/new-db db)
   :producer (producer/producer producer)))

(defn cli-config [env]
  {:db (config/db env)
   :producer (config/kafka env)})

(defn- cli-models
  "Returns the models for the client."
  [db models]
  (if-let [models (not-empty (split-by-comma models))]
    (models/by-names db models)
    (models/all db)))

(defn run
  "Run the client of the weather model worker."
  [{:keys [producer db] :as system} & args]
  (cli/with-commandline [[opts variables] args]
    [[h help "Print this help."]
     [m models "The weather models to load." :string "MODELS"]]
    (if (:help opts)
      (cli/print-help "weather-models [OPTION...]")
      (let [topic (-> (config) :input :commands)]
        (doseq [model (cli-models db (:models opts))]
          @(producer/produce! producer topic nil model)
          (log/info {:msg "Queued weather model to update inventory."
                     :model model}))))))

(defn -main
  "Publish commands to update the data sources of the weather models."
  [& args]
  (p/with-start [system (client (cli-config env))]
    (apply run system args)))

(comment
  (-main "-m" "gfs")
  (-main "-m" "nww3")
  (-main "-m" "akw")
  (-main))
