(ns burningswell.worker.users.profile-photos
  (:require [burningswell.db.emails :as emails]
            [burningswell.db.photos :as db-photos]
            [burningswell.db.users :as users]
            [burningswell.services.facebook :as facebook]
            [burningswell.services.gravatar :as gravatar]
            [burningswell.services.photos :as photos]
            [burningswell.worker.api.events :as events]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.topics :as topics]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [jackdaw.streams :as j]))

(def topics
  "The topics used by the user profile photo app."
  {:events topics/api-events-edn
   :created topics/user-profile-photo-created})

(defn config
  "Returns the configuration for the app."
  [& [opts]]
  (->> {:application
        {"application.id" "user-profile-photos"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"}
        :input {:events (:events topics)}
        :output {:created (:created topics)}}
       (merge opts)))

(defn- events [env builder]
  (j/kstream builder (-> env :config :input :events)))

(defn- user-created
  "Returns a stream of user-created event."
  [events]
  (events/by-name events :burningswell.api.events/user-created))

(defn- oauth-callback-succeeded
  "Returns a stream of OAuth callback event."
  [events]
  (events/by-name events :burningswell.api.events.oauth/callback-succeeded))

(defn- photo-title
  "Returns the profile photo title for `provider`."
  [provider]
  (str (str/capitalize (name provider)) " Profile Photo"))

(defn gravatar-photos
  "Save the Gravatar profile photos for a create user `event`."
  [{:keys [db gravatar photos] :as env} event]
  (if-let [user (users/by-id db (:user-id event))]
    (for [email (emails/by-user db user)
          :let [{:keys [thumbnail-url]} (gravatar/profile gravatar email)]
          :when (empty? (db-photos/by-user-and-url db user thumbnail-url))]
      {:title (photo-title :gravatar)
       :source-url thumbnail-url
       :user-id (:id user)})
    []))

(defmulti profile-photo-urls
  "Returns the profile photo urls from an OAuth callback event."
  (fn [env event] (-> event :provider keyword)))

(defmethod profile-photo-urls :facebook
  [{:keys [db facebook]} {:keys [user-id profile]}]
  (let [response (facebook/profile-photo facebook profile)]
    (when (not (:is-silhouette response))
      (-> response :url vector))))

(defmethod profile-photo-urls :google [_ event]
  (when (not (-> event :profile :image :is-default))
    (some-> event :profile :image :url vector)))

(defmethod profile-photo-urls :linkedin [_ event]
  (some-> event :profile :picture-url vector))

(defmethod profile-photo-urls :twitter [_ {:keys [profile]}]
  (when (not (:default-profile-image profile))
    (some-> (or (:profile-image-url-https profile)
                (:profile-image-url-http profile))
            vector)))

(defn oauth-provider-photos
  "Returns the profile photos from an OAuth callback event."
  [{:keys [db photos] :as env} {:keys [provider] :as event}]
  (when-let [user (users/by-id db (:user-id event))]
    (doall (for [url (profile-photo-urls env event)]
             (when (empty? (db-photos/by-user-and-url db user url))
               {:title (photo-title provider)
                :source-url url
                :user-id (:id user)})))))

(defn save-profile-photo!
  "Save the profile `photo` to the database."
  [{:keys [db photos] :as env} photo]
  (try (when-let [user (users/by-id db (:user-id photo))]
         (let [photo (photos/create-profile-photo! photos user photo)]
           (log/infof "Saved profile photo for user %s: %s."
                      (:user-id photo) (:url photo))
           photo))
       (catch Throwable e
         (case (-> e ex-data :type)
           ::photos/download-error nil
           (throw e)))))

(defn build-topology
  "Build the Kafka Streams topology."
  [app builder]
  (let [events (events app builder)]
    (-> (user-created events)
        (j/flat-map-values #(gravatar-photos app %))
        (j/map-values #(save-profile-photo! app %))
        (j/to (-> app :config :output :created)))
    (-> (oauth-callback-succeeded events)
        (j/flat-map-values #(oauth-provider-photos app %))
        (j/map-values #(save-profile-photo! app %))
        (j/to (-> app :config :output :created)))))

(defrecord Worker [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn worker [& [opts]]
  (-> (map->Worker {:config (config opts)})
      (component/using [:db :driver :gravatar :facebook :photos])))
