(ns burningswell.worker.geocoder
  (:require [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.serdes :as serdes]
            [burningswell.worker.topics :as topics :refer [deftopic]]
            [burningswell.worker.util :as util]
            [com.stuartsierra.component :as component]
            [geo.postgis :refer [point]]
            [geocoder.google :as geocoder]
            [inflections.core :as infl]
            [jackdaw.streams :as j]
            [taoensso.timbre :as log]))

(deftopic addresses-command-topic
  "burningswell.worker.geocoder.commands.addresses")

(deftopic addresses-event-topic
  "burningswell.worker.geocoder.events.addresses")

(deftopic addresses-table-topic
  "burningswell.worker.geocoder.tables.addresses")

(deftopic locations-command-topic
  "burningswell.worker.geocoder.commands.locations")

(deftopic locations-event-topic
  "burningswell.worker.geocoder.events.locations")

(deftopic locations-table-topic
  "burningswell.worker.geocoder.tables.locations")

(defn config
  "Returns the config for the geocoder app."
  [& [opts]]
  (->> {:application
        {"application.id" "burningswell.worker.geocoder"
         "bootstrap.servers" (:bootstrap.servers opts)}
        :input {:addresses addresses-command-topic
                :locations locations-command-topic}
        :output {:addresses addresses-event-topic
                 :locations locations-event-topic}
        :tables {:addresses addresses-table-topic
                 :locations locations-table-topic}}
       (merge opts)))

(defn- lookup-country
  "Lookup the country for `result`."
  [{:keys [db] :as app} result]
  (let [iso-3166-1-alpha-2 (-> result geocoder/country :iso-3166-1-alpha-2)
        location (-> result geocoder/location util/location->point)
        country (or (countries/by-iso-3166-1-alpha-2 db iso-3166-1-alpha-2)
                    (first (countries/within-distance db location 1)))]
    (if country
      (assoc result :country (select-keys country [:id :name]))
      result)))

(defn- lookup-region
  "Lookup the region for `result`."
  [{:keys [db] :as app} result]
  (let [point (util/location->point (geocoder/location result))
        region (first (regions/within-distance db point 1))]
    (if region
      (assoc result :region (select-keys region [:id :name]))
      result)))

(defn- extend-result
  "Extend the `result` with country and region."
  [app result]
  (->> result
       (lookup-region app)
       (lookup-country app)))

(defn- extend-results
  "Extend the :results of `event` with country and region."
  [app results]
  (mapv #(extend-result app %) results))

(defn- geocode-address!
  "Geocode the given `address` or throw an exception on error."
  [app address & [opts]]
  (geocoder/geocode-address (:geocoder app) address))

(defn- geocode-location!
  "Geocode the given `location` or throw an exception on error."
  [app location & [opts]]
  (geocoder/geocode-location (:geocoder app) location))

(defn- log-result
  "Log the geocoder `result`."
  [result]
  (let [{:keys [lat lng]} (geocoder/location result)]
    (log/infof "  - %s, %s, %s" (:formatted-address result) lat lng)))

(defn- num-results-str
  [results]
  (infl/pluralize (count results) "result"))

(defn- log-address-success
  "Log the address geocode result."
  [address results]
  (log/infof "Geocoded address: %s (%s)" address (num-results-str results))
  (dorun (map log-result results)))

(defn- log-location-success
  "Log the location geocode result."
  [location results]
  (log/infof "Geocoded location: %s, %s (%s)" (:lat location) (:lng location)
             (num-results-str results))
  (dorun (map log-result results)))

(defn- address-error
  "Returns a address geocode error."
  [ex address opts]
  (let [{:keys [type] :as data} (ex-data ex)]
    {:status :error
     :op :geocode-address
     :address address
     :data data
     :msg (.getMessage ex)
     :opts opts
     :type type}))

(defn- address-success
  "Returns the address geocode success event."
  [address opts results]
  {:status :ok
   :op :geocode-address
   :address address
   :opts opts
   :results results})

(defn geocode-address
  "Geocode the given `address`."
  [app address & [opts]]
  (try (let [results (geocode-address! app address opts)
             results (extend-results app results)]
         (log-address-success address results)
         (address-success address opts results))
       (catch Exception ex
         (log/errorf ex "Can't geocode address: %s." address)
         (address-error ex address opts))))

(defn- location-error
  "Returns a location geocode error."
  [ex location opts]
  (let [{:keys [type] :as data} (ex-data ex)]
    {:status :error
     :op :geocode-location
     :data (select-keys data [:status :body :headers])
     :location location
     :msg (.getMessage ex)
     :opts opts
     :type type}))

(defn location-success
  "Returns the location geocode success event."
  [location opts results]
  {:status :ok
   :op :geocode-location
   :location location
   :opts opts
   :results results})

(defn geocode-location
  "Geocode the given `location`."
  [app location & [opts]]
  (try (let [results (geocode-location! app location opts)
             results (extend-results app results)]
         (log-location-success location results)
         (location-success location opts results))
       (catch Exception ex
         (log/errorf ex "Can't geocode location: %s" (pr-str location))
         (location-error ex location opts))))

(defn- build-addresses-topology
  "Build the geocoder addresses topology."
  [app builder]
  (-> (j/kstream builder (-> app :config :input :addresses))
      (j/map-values #(geocode-address app (:address %)))
      (j/map (fn [[k v]] [(:address v) v]))
      (j/group-by (fn [[_ event]] (:address event))
                  {:key-serde (serdes/edn)
                   :value-serde (serdes/edn)})
      (j/reduce (fn [_ event] event) (-> app :config :tables :addresses))
      (j/to-kstream)
      (j/to (-> app :config :output :addresses))))

(defn- build-locations-topology
  "Build the geocoder locations topology."
  [app builder]
  (-> (j/kstream builder (-> app :config :input :locations))
      (j/map-values #(geocode-location app (:location %)))
      (j/map (fn [[k v]] [(:location v) v]))
      (j/group-by (fn [[_ event]] (:location event))
                  {:key-serde (serdes/edn)
                   :value-serde (serdes/edn)})
      (j/reduce (fn [_ event] event) (-> app :config :tables :locations))
      (j/to-kstream)
      (j/to (-> app :config :output :locations))))

(defn- build-topology
  "Build the geocoder topology."
  [app builder]
  (build-addresses-topology app builder)
  (build-locations-topology app builder))

(defrecord Geocoder [config geocoder driver]
  component/Lifecycle
  (start [app]
    (->> (geocoder/geocoder config)
         (assoc app :geocoder)
         (driver/start driver)))

  (stop [app]
    (-> (driver/stop driver app)
        (assoc :geocoder nil)))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn service
  "Returns a new geocoder worker for `opts`."
  [& [opts]]
  (-> (map->Geocoder {:config (config opts)})
      (component/using [:db :driver])))
