(ns burningswell.services.facebook
  (:require [burningswell.specs.core :as specs]
            [clj-http.client :as http]
            [clojure.spec.alpha :as s]
            [inflections.core :as infl]))

(s/def ::url ::specs/url-str)
(s/def ::version string?)

(s/def ::service
  (s/keys :req-un [::url ::version]))

(def defaults
  "The default Facebook config."
  {:url "https://graph.facebook.com"
   :version "3.2"})

(defn root-url
  "Returns the Graph root url of `service`."
  [service]
  (format "%s/v%s/" (:url service) (:version service)))

(s/fdef root-url
  :args (s/cat :service ::service)
  :ret ::specs/url-str)

(defn profile-photo
  "Fetch the Facebook `profile` photo."
  [service profile]
  (try (-> (http/get (str (root-url service) "/" (:id profile) "/picture")
                     {:as :json
                      :query-params
                      {:redirect false
                       :type "large"}})
           :body infl/hyphenate-keys :data)
       (catch Throwable e
         (case (-> e ex-data :status)
           404 nil
           (throw (ex-info (str "Can't fetch Facebook profile photo. "
                                (.getMessage e))
                           {:profile profile} e))))))

(s/fdef profile-photo
  :args (s/cat :service ::service
               :profile (s/keys :req-un [:facebook.profile/id])))

(defrecord Facebook [url version])

(defn service
  "Returns the Facebook service."
  [& [opts]]
  (map->Facebook (merge defaults opts)))

(s/fdef service
  :args (s/cat :opts (s/? (s/nilable map?)))
  :ret ::service)
