(ns burningswell.worker.weather.datasets
  (:gen-class)
  (:require [burningswell.config.core :as config]
            [burningswell.db.connection :as db]
            [burningswell.db.weather.datasets :as datasets]
            [burningswell.db.weather.datasources :as datasources]
            [burningswell.db.weather.models :as models]
            [burningswell.db.weather.variables :as variables]
            [burningswell.io :refer [slurp-byte-array]]
            [burningswell.services.storage :as storage]
            [burningswell.shell :refer [sh!]]
            [burningswell.worker.driver :as driver]
            [burningswell.worker.producer :as producer]
            [burningswell.worker.topics :as topics]
            [burningswell.worker.weather.gdal :as gdal]
            [burningswell.time :as time]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.core :as t]
            [clojure.java.io :as io]
            [clojure.pprint :as pprint]
            [clojure.spec.alpha :as s]
            [com.stuartsierra.component :as component]
            [commandline.core :as cli]
            [datumbazo.core :as sql]
            [datumbazo.shell :as shell :refer [raster2pgsql]]
            [datumbazo.util :refer [exec-sql-file]]
            [environ.core :refer [env]]
            [jackdaw.streams :as j]
            [netcdf.dataset :as dataset]
            [no.en.core :refer [split-by-comma]]
            [peripheral.core :as p]
            [taoensso.timbre :as log]))

(defn config
  "Returns the configuration for the app."
  [& [opts]]
  (->> {:application
        {"application.id" "update-weather-datasets"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"
         "num.stream.threads" "2"}
        :input {:commands topics/weather-dataset-update}
        :output {:succeeded topics/weather-dataset-update-succeeded}}
       (merge opts)))

(defn dataset-file
  "Return the file of the `dataset` with the given `extension`."
  [{:keys [directory datasource variable model valid-time]
    :as dataset} extension]
  (let [valid-time (to-date-time valid-time)]
    (str (io/file
          (or directory "")
          (:name model)
          (:name variable)
          (format "%04d" (t/year valid-time))
          (format "%02d" (t/month valid-time))
          (format "%02d" (t/day valid-time))
          (format "%02d" (t/hour valid-time))
          (str (to-date-time (:reference-time datasource))
               "." extension)))))

(s/fdef dataset-file
  :args (s/cat :dataset ::datasets/dataset :extension string?)
  :ret string?)

(defn raster-table
  "Returns the temporary raster table for `dataset`."
  [{:keys [variable model valid-time] :as dataset}]
  (let [valid-time (to-date-time valid-time)]
    (keyword (format "%s_%s_%04d_%02d_%02d_%02d"
                     (:name model) (:name variable)
                     (t/year valid-time)
                     (t/month valid-time)
                     (t/day valid-time)
                     (t/hour valid-time)))))

(s/fdef raster-table
  :args (s/cat :dataset ::datasets/dataset)
  :ret keyword?)

(defn load-dataset
  "Load the `dataset` and it's dependencies from `db`."
  [db {:keys [directory] :as dataset}]
  (let [datasource (datasources/by-id db (:datasource-id dataset))
        directory (or directory "data")
        model (models/by-id db (:model-id datasource))
        variable (variables/by-id db (:variable-id dataset))]
    (->> {:datasource datasource
          :directory directory
          :ingest-started-at (t/now)
          :model model
          :variable variable}
         (merge dataset))))

(s/fdef load-dataset
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn center-dataset!
  "Center the dataset."
  [{:keys [filename] :as dataset}]
  (let [vrt-file (dataset-file dataset "center.vrt")]
    (gdal/create-virtual-format filename vrt-file {:center? true})
    (assoc dataset :filename vrt-file :vrt-file vrt-file)))

(s/fdef center-dataset!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn- download-msg
  "Returns the download log message."
  [started-at]
  (format "Downloaded dataset in %s."
          (time/format-interval (t/interval started-at (t/now)))))

(defn download-dataset!
  "Download the dataset and it's dependencies from `dataset`."
  [{:keys [valid-time] :as dataset}]
  (let [started-at (t/now)
        url (-> dataset :datasource :dods)
        variable (-> dataset :variable :name)
        raster-file (dataset-file dataset "source.geotiff")
        checksum-file (dataset-file dataset "source.geotiff.md5")]
    (when-not (.exists (io/file raster-file))
      (dataset/with-grid-dataset [grid url]
        (dataset/write-geotiff grid variable valid-time raster-file))
      (log/info {:msg (download-msg started-at)
                 :dataset dataset}))
    (prn "YO" (time/format-interval (t/interval started-at (t/now))))
    (->> {:download-started-at started-at
          :download-finished-at (t/now)
          :filename raster-file
          :raster-file raster-file
          :checksum-file checksum-file}
         (merge dataset))))

(s/fdef download-dataset!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn read-gdal-info!
  "Read the GDAL info of `dataset` and add it to :gdal-info."
  [{:keys [filename] :as dataset}]
  (assoc dataset :gdal-info (gdal/info filename)))

(s/fdef read-gdal-info!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn set-no-data-value!
  "Set the no-data value of `dataset` to the computed minimum from GDAL info."
  [{:keys [filename gdal-info] :as dataset}]
  (let [no-data-file (dataset-file dataset "no-data.vrt")
        no-data-value (-> gdal-info :bands first :computedMin)]
    (assert no-data-value)
    (sh! "gdal_translate" "-a_nodata" (str no-data-value) (str filename)
         (str no-data-file))
    (assoc dataset :filename no-data-file :no-data-file no-data-file)))

(s/fdef set-no-data-value!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn reproject-dataset!
  "Project `dataset` into the Mercator projection."
  [{:keys [filename] :as dataset}]
  (let [reproject-file (dataset-file dataset "mercator.vrt")]
    (gdal/to-mercator filename reproject-file)
    (assoc dataset :filename reproject-file :reproject-file reproject-file)))

(s/fdef reproject-dataset!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn convert-dataset-to-sql!
  "Download the dataset and it's dependencies from `dataset`."
  [db {:keys [filename] :as dataset}]
  (let [sql-file (dataset-file dataset "sql")
        raster-table (raster-table dataset)]
    (raster2pgsql db raster-table filename sql-file
                  {:constraints true
                   :height 100
                   :mode :create
                   :no-transaction true
                   :padding true
                   :regular-blocking true
                   :width 100})
    (->> {:sql-file sql-file
          :raster-table raster-table}
         (merge dataset))))

(s/fdef convert-dataset-to-sql!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn convert-dataset-to-geotif!
  "Download the dataset and it's dependencies from `dataset`."
  [{:keys [filename] :as dataset}]
  (let [tif-file (dataset-file dataset "tif")]
    (sh! "gdal_translate"
         "-co" "COMPRESS=LZW"
         (str filename) (str tif-file))
    (merge dataset {:filename tif-file :tif-file tif-file})))

(s/fdef convert-dataset-to-geotif!
  :args (s/cat :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn import-dataset!
  "Download the dataset and it's dependencies from `dataset`."
  [db {:keys [sql-file raster-table] :as dataset}]
  @(sql/drop-table db [raster-table]
     (sql/if-exists true))
  (exec-sql-file db sql-file)
  dataset)

(s/fdef import-dataset!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn delete-dataset!
  "Delete the older data for the `dataset`."
  [db {:keys [model variable valid-time partition-table] :as dataset}]
  @(sql/delete db partition-table
     (sql/where `(and (= :model-id ~(:id model))
                      (= :variable-id ~(:id variable))
                      (= :valid-time ~valid-time))))
  dataset)

(s/fdef delete-dataset
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn insert-dataset!
  [db {:keys [datasource model raster-table valid-time
              variable partition-table] :as dataset}]
  (let [column (keyword (str (name raster-table) ".rast"))
        columns [:model-id :variable-id :dataset-id
                 :reference-time :valid-time :rast]]
    (->> @(sql/insert db partition-table columns
            (sql/select db [(sql/as (:id model) :model-id)
                            (sql/as (:id variable) :variable-id)
                            (sql/as (:id dataset) :dataset-id)
                            (sql/as (:reference-time datasource) :reference-time)
                            (sql/as valid-time :valid-time)
                            `(st_band ~column 1)]
              (sql/from raster-table))
            (sql/returning :id))
         first :id (assoc dataset :id))))

(s/fdef insert-dataset!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset))

(defn- partition-table
  "Returns the partition table name."
  [valid-time]
  (let [valid-time (to-date-time valid-time)]
    (keyword (format "weather.rasters-%04d-%02d"
                     (t/year valid-time)
                     (t/month valid-time)))))

(s/fdef partition-table
  :args (s/cat :valid-time inst?)
  :ret keyword?)

(defn- partition-start-time
  "Returns the partition start time for `valid-time`."
  [valid-time]
  (-> valid-time (.withDayOfMonth 1) (.withTime 0 0 0 0)))

(defn- partition-end-time
  "Returns the partition end time for `valid-time`."
  [valid-time]
  (t/plus (partition-start-time valid-time) (t/months 1)))

(defn create-dataset-partition!
  "Create the raster partition table for the `dataset` in `db`."
  [db {:keys [valid-time] :as dataset}]
  (let [partition (partition-table valid-time)
        valid-time (to-date-time valid-time)
        start-time (partition-start-time valid-time)
        end-time (partition-end-time valid-time)]
    @(sql/create-table db partition
       (sql/if-not-exists true)
       (sql/check `(and (>= :valid-time (cast ~start-time :timestamptz))
                        (< :valid-time (cast ~end-time :timestamptz))))
       (sql/like :weather.rasters :including [:all])
       (sql/inherits :weather.rasters))
    (assoc dataset :partition-table partition)))

(s/fdef create-dataset-partition!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn save-dataset-to-db!
  "Save the `dataset` to `db`."
  [db dataset]
  (let [dataset (create-dataset-partition! db dataset)]
    (sql/with-transaction [db db]
      (->> (import-dataset! db dataset)
           (delete-dataset! db)
           (insert-dataset! db))
      dataset)))

(s/fdef save-dataset-to-db!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn dataset-complete!
  "Save the meta data of `dataset` to `db`."
  [db {:keys [datasource model variable] :as dataset}]
  (let [dataset (assoc dataset :ingest-finished-at (t/now))]
    @(sql/update db :weather.datasets
       (select-keys dataset [:geotiff-url
                             :ingest-started-at
                             :ingest-finished-at])
       (sql/where `(= :id ~(:id dataset)))
       (sql/returning :*))
    (log/info {:msg "Updated weather dataset."
               :dataset (select-keys dataset [:id])
               :datasource (select-keys datasource [:id :reference-time])
               :model (select-keys model [:id :name])
               :variable (select-keys variable [:id :name])})
    dataset))

(s/fdef dataset-complete!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn save-dataset-to-storage!
  "Save the `dataset` to `storage`."
  [storage {:keys [model variable valid-time] :as dataset}]
  (let [filename (-> dataset :filename io/file .getName)
        key (dataset-file (assoc dataset :directory "weather") "tif")
        bytes (slurp-byte-array (:filename dataset))
        {:keys [url]}
        (->> {:acl storage/acl-public
              :bytes bytes
              :cache-control "public, max-age 31536000"
              :content-disposition (format "attachment; filename=%s" filename)
              :content-encoding :identity
              :key key
              :meta-data
              {:dataset-id (:id dataset)
               :model-id (:id model)
               :model-name (:name model)
               :valid-time (str (to-date-time valid-time))
               :variable-id (:id variable)
               :variable-name (:name variable)}}
             (storage/save! storage))]
    (assoc dataset :geotiff-url url)))

(s/fdef save-dataset-to-storage!
  :args (s/cat :storage any? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn cleanup-files
  "Returns the cleanup files from `dataset`."
  [dataset]
  (map dataset [:checksum-file
                :no-data-file
                :raster-file
                :reproject-file
                :vrt-file
                :sql-file
                :tif-file]))

(s/fdef cleanup-files
  :args (s/cat :dataset ::datasets/dataset))

(defn cleanup-dataset!
  "Cleanup the `dataset` into `db`"
  [db {:keys [raster-table] :as dataset}]
  @(sql/drop-table db [raster-table])
  (doseq [file (cleanup-files dataset)]
    (.delete (io/file file)))
  dataset)

(s/fdef cleanup-dataset!
  :args (s/cat :db sql/db? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

(defn update-dataset!
  "Download the `dataset`, convert and center it, and finally load it
  into the `db`."
  [{:keys [db storage]} dataset]
  (->> (load-dataset db dataset)
       (download-dataset!)
       (read-gdal-info!)
       (set-no-data-value!)
       (center-dataset!)
       (reproject-dataset!)
       (convert-dataset-to-geotif!)
       (convert-dataset-to-sql! db)
       (save-dataset-to-storage! storage)
       (save-dataset-to-db! db)
       (cleanup-dataset! db)
       (dataset-complete! db)))

(s/fdef update-dataset!
  :args (s/cat :worker map? :dataset ::datasets/dataset)
  :ret ::datasets/dataset)

;; Kafka Streams

(defn build-topology
  "Build the Kafka Streams topology."
  [{:keys [db config storage] :as app} builder]
  (-> (j/kstream builder (-> config :input :commands))
      (j/map-values #(update-dataset! app %))
      (j/to (-> config :output :succeeded))))

(defrecord Worker [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn worker
  "Returns a new weather model worker."
  [& [opts]]
  (-> (map->Worker {:config (config opts)})
      (component/using [:db :driver :storage])))

;; Command line client

(defn client
  "Returns a client for the weather dataset worker."
  [{:keys [db producer] :as config}]
  (component/system-map
   :db (db/new-db db)
   :producer (producer/producer producer)))

(defn cli-config [env]
  {:db (config/db env)
   :producer (config/kafka env)})

(defn- cli-models
  "Returns the models for the client."
  [db models]
  (if-let [models (not-empty (split-by-comma models))]
    (models/by-names db models)
    (models/all db)))

(defn- cli-variables
  "Returns the variables for the client."
  [db variables]
  (if-let [variables (not-empty (split-by-comma variables))]
    (variables/by-names db variables)
    (variables/all db)))

(defn- cli-datasets
  "Returns the datasets for the client."
  [db models variables start end]
  @(sql/select db [:datasources.model-id
                   :datasources.reference-time
                   :datasets.*]
     (sql/from :weather.datasets)
     (sql/join :weather.datasources.id
               :weather.datasets.datasource-id)
     (sql/join :weather.recent-datasets.id :weather.datasets.id)
     (sql/where `(is-null :datasets.ingest-finished-at))
     (when start
       (sql/where `(>= :datasets.valid-time ~start) :and))
     (when end
       (sql/where `(< :datasets.valid-time ~end) :and))
     (when (seq models)
       (sql/where `(in :datasources.model-id ~(map :id models)) :and))
     (when (seq variables)
       (sql/where `(in :datasets.variable-id ~(map :id variables)) :and))
     (sql/order-by :datasources.model-id
                   :datasets.variable-id
                   :datasets.valid-time)))

(def dataset-table-keys
  [:id :datasource-id :model-id :variable-id :valid-time :reference-time])

(defn- pprint-datasets [datasets]
  (println )
  (pprint/print-table dataset-table-keys datasets))

(defn run
  "Run the client of the weather dataset worker."
  [{:keys [db producer] :as system} & args]
  (cli/with-commandline [[opts variables] args]
    [[h help "Print this help."]
     [m models "The weather models to load." :string "MODELS"]
     [v variables "The weather variables to load." :string "VARIABLES"]
     [s start "The start time." :time "START"]
     [e end "The end time." :time "END"]]
    (if (:help opts)
      (cli/print-help "weather-datasets [OPTION...]")
      (let [start (or (to-date-time (:start opts))
                      (t/minus (t/now) (t/hours 3)))
            end (to-date-time (:end opts))
            models (cli-models db (:models opts))
            variables (cli-variables db (:variables opts))
            datasets (cli-datasets db models variables start end)
            topic (-> (config) :input :commands)]
        (doseq [dataset datasets]
          @(producer/produce! producer topic nil dataset))
        (log/info {:datasets (count datasets)
                   :end end
                   :models (mapv :name models)
                   :msg "Queued weather datasets for loading."
                   :start start
                   :variables (mapv :name variables)})
        (pprint-datasets datasets)
        datasets))))

(defn -main [& args]
  (p/with-start [system (client (cli-config env))]
    (apply run system args)
    nil))

(comment
  (-main "-m" "nww3" "-v" "htsgwsfc,wvdirsfc")
  (-main "-m" "gfs" "-v" "tmpsfc")
  (-main "-m" "nww3")

  (-main "-e" "2018-12-29" "-m" "nww3" "-v" "htsgwsfc")
  (-main "-s" "2018-04-24" "-e" "2018-04-25" "-m" "nww3,gfs" "-v" "htsgwsfc,tmpsfc")
  (-main "-s" "2017-12-14")
  (-main "-m" "nww3,gfs" "-v" "htsgwsfc,tmpsfc")
  (-main "-m" "nww3,gfs")
  (-main "-s" "2017-12-12" "-m" "nww3")
  (-main "-s" "2017-12-12" "-m" "nww3" "-v" "wvdirsfc"))
