(ns burningswell.worker.page-views
  (:require [burningswell.worker.driver :as driver]
            [burningswell.api.specs.events]
            [burningswell.worker.topics :as topics]
            [burningswell.routes :as routes]
            [com.stuartsierra.component :as component]
            [jackdaw.serdes.edn :as jse]
            [jackdaw.streams :as j]
            [no.en.core :refer [format-url parse-url]]))

(def topics
  "The topics used by the page views app."
  {:events (topics/config "burningswell.api.events")
   :total (topics/config "burningswell.page-views.urls.total")
   :total-per-user (topics/config "burningswell.page-views.urls.total-per-user")})

(defn config
  "Returns the config for the page views app."
  [& [opts]]
  (->> {:application
        {"application.id" "url-views"
         "bootstrap.servers" (:bootstrap.servers opts)
         "cache.max.bytes.buffering" "0"}
        :input {:events (:events topics)}
        :output {:total (:total topics)
                 :total-per-user (:total-per-user topics)}}
       (merge opts)))

(defn- strip-url
  "Strip the query params from `url`."
  [url]
  (some-> (parse-url url)
          (dissoc :query-params)
          (format-url)
          (java.net.URL.)))

(defn- match-route
  "Returns the matching route for `url`, or nil. "
  [url]
  (some-> url parse-url :uri routes/match (dissoc :uri)))

(defn total-key
  "Returns the key for the total page counts."
  [[key {:keys [url]}]]
  {:route (match-route url)
   :url (strip-url url)})

(defn total-per-user-key
  "Returns the key for the total page counts per user."
  [[key {:keys [url user-id]}]]
  {:route (match-route url)
   :url (strip-url url)
   :user-id user-id})

(defn- authenticated?
  "Returns true if the page view `event` is from an authenticated user,
  otherwise false."
  [[key event]]
  (some? (:user-id event)))

(defn- page-view-event?
  "Returns true if the `event` is a page view event, otherwise false."
  [[key event]]
  (= (:name event) :burningswell.api.events/page-view-created))

(defn- page-view-events
  "Build a stream of page view events."
  [app builder]
  (-> (j/kstream builder (-> app :config :input :events))
      (j/filter page-view-event?)))

(defn- total
  "Build a table of total page views."
  [app events]
  (-> (j/group-by events total-key (topics/config nil))
      (j/count (-> app :config :output :total))))

(defn- total-per-user
  "Build a table of total page views per user."
  [app events]
  (-> (j/filter events authenticated?)
      (j/group-by total-per-user-key (topics/config nil))
      (j/count (-> app :config :output :total-per-user))))

(defn- build-topology
  "Build the page view topology."
  [app builder]
  (let [page-views (page-view-events app builder)]
    (-> (total app page-views)
        (j/to-kstream)
        (j/to (-> app :config :output :total)))
    (-> (total-per-user app page-views)
        (j/to-kstream)
        (j/to (-> app :config :output :total-per-user)))))

(defrecord PageViews [config driver]
  component/Lifecycle
  (start [app]
    (driver/start driver app))

  (stop [app]
    (driver/stop driver app))

  driver/Application
  (config [app]
    (:application config))

  (topology [app builder]
    (build-topology app builder)))

(defn page-views [& [opts]]
  (-> (map->PageViews {:config (config opts)})
      (component/using [:driver])))
