(ns burningswell.worker.photos
  (:require [burningswell.db.images :as images]
            [burningswell.db.photos :as photos]
            [burningswell.rabbitmq.client :as rabbitmq]
            [burningswell.worker.storage :as storage]
            [clj-http.client :as http]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [image-resizer.core :as resizer]
            [kithara.core :as k]
            [kithara.patterns.dead-letter-backoff
             :refer [with-durable-dead-letter-backoff]]
            [kithara.rabbitmq.queue :as queue]
            [pandect.core :as digest]
            [ring.util.codec :refer [base64-encode]]
            [slingshot.slingshot :refer [try+]])
  (:import [com.google.cloud.storage Acl Acl$User Acl$Role BlobInfo BucketInfo
            StorageOptions Storage$BlobTargetOption]
           java.awt.image.BufferedImage
           [org.apache.commons.io IOUtils]
           [java.io ByteArrayInputStream ByteArrayOutputStream]
           javax.imageio.ImageIO))

(def queue
  "The photos worker queue."
  {:auto-delete? false
   :durable? true
   :exclusive? false
   :name "worker.photos"})

(defn setup-topology
  "Setup the topology of the photo worker."
  [channel]
  (-> (rabbitmq/declare-queue channel queue)
      (queue/bind {:exchange "api" :routing-keys ["photos.created"]}))  )

(defn- buffered-image->bytes
  "Convert `buffered-image` into a byte in `format`."
  [^BufferedImage buffered-image ^String format]
  (let [baos (ByteArrayOutputStream.)]
    (assert (ImageIO/write buffered-image format baos)
            (str "Can't write image in format: " format))
    (.toByteArray baos)))

(defn- content-md5
  "Return the MD5 sum of `bytes`."
  [bytes]
  (base64-encode (digest/md5-bytes bytes)))

(defn- content-type
  "Return the content type of `response`."
  [response]
  (get-in response [:headers "content-type"]))

(defn content-type-extension
  "Returns the filename extension from `content-type`."
  [content-type]
  (if-not (str/blank? content-type)
    (last (str/split content-type #"/"))))

(defn- content-disposition
  "Return the content disposition header of `photo`."
  [photo label content-type]
  (str "attachment; filename=photo-"
       (:id photo) "-" label "."
       (content-type-extension content-type)))

(defn image-storage-key
  "Return the storage key for `photo`, `label` and `content-type`."
  [worker photo label content-type]
  (let [extension (content-type-extension content-type)]
    (str "photos" "/" (:id photo) "/" label "." extension)))

(defn image-storage-url
  "Return the storage url for `photo`, `label` and `content-type`."
  [worker photo label content-type]
  (format "https://storage.googleapis.com/%s/%s"
          (-> worker :google :storage :bucket)
          (image-storage-key worker photo label content-type)))

(defn download-image
  "Download the original image of `photo`."
  [worker {:keys [id] :as photo}]
  (let [response (http/get (:url photo) {:as :stream})
        bytes (IOUtils/toByteArray (:body response)) ]
    (log/infof "Downloaded photo %s." id)
    (if-let [image (ImageIO/read (ByteArrayInputStream. bytes))]
      (let [content-type (content-type response)]
        (log/infof "Decoded photo %s as content type %s." id content-type)
        {:buffered image
         :bytes bytes
         :content-type content-type
         :label "original"})
      (throw (ex-info "Can't decode photo."
                      {:type :decode-error
                       :photo photo
                       :response response})))))

(defn save-image-to-db
  "Save the `image` to the database."
  [worker photo image]
  (let [{:keys [buffered content-type label]} image]
    (->> {:photo-id (:id photo)
          :content-disposition (content-disposition photo label content-type)
          :content-length (count (:bytes image))
          :content-md5 (content-md5 (:bytes image))
          :height (.getHeight buffered)
          :storage-key (image-storage-key worker photo label content-type)
          :url (image-storage-url worker photo label content-type)
          :width (.getWidth buffered)}
         (merge image)
         (images/save! (:db worker))
         (merge image))))

(defn save-image-to-storage
  "Upload the `image`."
  [worker photo image]
  (->> {:acl storage/acl-public
        :bytes (:bytes image)
        :cache-control "public, max-age 31536000"
        :content-disposition (:content-disposition image)
        :content-encoding :identity
        :key (:storage-key image)
        :meta-data {:id (str (:id image))
                    :photo-id (str (:id photo))
                    :label (:label image)
                    :width (:width image)
                    :height (:height image)}}
       (storage/save! (:storage worker)))
  (log/infof "Saved image #%d to Google Storage at %s."
             (:id image) (:url image))
  image)

(defn resize-image
  "Resize `image` to `dimension`."
  [image dimension]
  (let [{:keys [label width height]} dimension
        buffered (:buffered image)
        resized (cond
                  (and width height)
                  (resizer/resize buffered width height)
                  width
                  (resizer/resize-to-width buffered width)
                  height
                  (resizer/resize-to-height buffered height))]
    {:buffered resized
     :bytes (buffered-image->bytes resized "jpg")
     :content-type "image/jpeg"
     :label label}))

(defn import-image
  "Import the `image` of `photo`."
  [worker photo image]
  (->> (save-image-to-db worker photo image)
       (save-image-to-storage worker photo)))

(defn download-and-import
  "Download the original image of `photo` and import it."
  [worker photo]
  (->> (download-image worker photo)
       (import-image worker photo)))

(defn resize-and-import
  "Resize the `original` image of `photo` to `dimension` and import it."
  [worker photo original dimension]
  (->> (resize-image original dimension)
       (import-image worker photo)))

(defn update-photo-status
  "Update the `status` of `photo` in the database."
  [worker photo status]
  (->> (assoc photo :status (name status))
       (photos/update! (:db worker))))

(defn import-photo
  "Import the `photo`."
  [worker photo]
  (try+
   (let [original (download-and-import worker photo)]
     (doseq [dimension (:dimensions worker)]
       (resize-and-import worker photo original dimension))
     (log/infof "Photo %s successfully imported." (:id photo))
     (update-photo-status worker photo :finished))
   (catch [:type :decode-error] _
     (log/errorf "Photo import error: Can't decode image." (:id photo))
     (update-photo-status worker photo :decode-error))
   (catch [:status 401] _
     (log/errorf "Photo import error: Image download forbidden." (:id photo))
     (update-photo-status worker photo :forbidden))
   (catch [:status 404] _
     (log/errorf "Photo import error: Image not found." (:id photo))
     (update-photo-status worker photo :not-found))))

(def dimensions
  "The dimension used by the photos worker to scale images."
  [{:label "tiny" :width 320}
   {:label "small" :width 480}
   {:label "medium" :width 640}
   {:label "large" :width 800}
   {:label "huge" :width 1024}])

(defmulti process-message
  "Handle messages that require geocoding of addresses."
  (fn [message] (-> message :routing-key keyword)))

(defmethod process-message :photos.created
  [{:keys [body env]}]
  (import-photo env body)
  {:status :ack})

(defmethod process-message :photos.deleted
  [{:keys [body env]}]
  {:status :ack})

(defmethod process-message :photos.updated
  [{:keys [body env]}]
  (import-photo env body)
  {:status :ack})

(defn worker
  "Return a new photo worker."
  [config]
  (-> process-message
      (k/consumer {:as rabbitmq/read-edn :consumer-name "Photo Worker"})
      (with-durable-dead-letter-backoff)
      (k/with-durable-queue "worker.photos"
        {:exchange "api"
         :routing-keys ["photos.created" "photos.deleted" "photos.updated"]})
      (k/with-channel {:prefetch-count 1})
      (k/with-connection (rabbitmq/config (:broker config)))
      (k/with-env (assoc config :dimensions dimensions))
      (component/using [:db :storage :topology])))

(comment

  (def storage (.. StorageOptions getDefaultInstance getService))

  (BucketInfo/of "burningswell-test")
  )
