(ns burningswell.worker.vision
  (:require [burningswell.io :as io]
            [burningswell.time :refer [current-duration-str]]
            [clj-time.core :as time]
            [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.tools.logging :as log]
            [inflections.core :as i])
  (:import [com.google.cloud.vision.v1 AnnotateImageRequest
            AnnotateImageResponse Feature Feature$Type
            Image ImageAnnotatorClient]
           [com.google.protobuf ByteString MessageLiteOrBuilder]))

(s/def ::description string?)

(s/def ::mid
  (s/with-gen string?
    #(gen/fmap (fn [id] (str "/m/" id)) (gen/not-empty (s/gen string?)))))

(s/def ::score
  (s/double-in :min 0.0 :max 1.0 :NaN? false :infinite? false))

(defn service?
  "Returns true if `x` is a Vision service, otherwise false."
  [x]
  (instance? ImageAnnotatorClient x))

(defn response?
  "Returns true if `x` is an annotated image response, otherwise false."
  [x]
  (instance? AnnotateImageResponse x))

(defn service []
  (ImageAnnotatorClient/create))

(defn feature
  "Return a `Feature` instance."
  [type & [max-results]]
  (-> (Feature/newBuilder)
      (.setType
       (case type
         :faces Feature$Type/FACE_DETECTION
         :labels Feature$Type/LABEL_DETECTION
         :landmarks Feature$Type/LANDMARK_DETECTION))
      (.setMaxResults (int (or max-results 5)))
      (.build)))

(s/fdef feature
  :args (s/cat :type #{:faces :labels :landmarks}
               :max-results (s/? pos-int?)))

(defn slurp-image [image]
  (if (bytes? image)
    image
    (let [bytes (io/slurp-byte-array image)]
      (-> (Image/newBuilder)
          (.setContent (ByteString/copyFrom bytes))
          (.build)))))

(defn- annotate-request
  "Return the annotate request for `image`."
  [image & [{:keys [faces labels landmarks]}]]
  (.build (cond-> (AnnotateImageRequest/newBuilder)
            faces (.addFeatures (feature :faces faces))
            image (.setImage (slurp-image image))
            labels (.addFeatures (feature :labels labels))
            landmarks (.addFeatures (feature :landmarks landmarks)))))

(defn- extract-message
  "Convert an entity annotation into a Clojure map."
  [annotation]
  (->> (for [[k v] (into {} (.getAllFields annotation))]
         [(-> k .getName i/hyphenate keyword)
          (cond
            (instance? java.util.Collection v)
            (mapv extract-message v )
            (instance? MessageLiteOrBuilder v)
            (extract-message v)
            :else v)])
       (into {})))

(defn- extract-annotations
  "Convert an entity annotation into a Clojure map."
  [response]
  (->> (for [[type annotation]
             [[:faces (.getFaceAnnotationsList response)]
              [:labels (.getLabelAnnotationsList response)]
              [:landmarks (.getLandmarkAnnotationsList response)]]
             :when (not (empty? annotation))]
         {type (mapv extract-message annotation)})
       (apply merge)))

(defn annotate-images
  "Annotate `images` using the Google Cloud Vision `service`."
  [service images & [opts]]
  (let [started-at (time/now)
        requests (map #(annotate-request %1 opts) images)
        batch-response (.batchAnnotateImages service requests)]
    ;; (.setDisableGZipContent annotate true)
    (let [responses (.getResponsesList batch-response)]
      (log/infof "Annotated batch of %s images in %s." (count images)
                 (current-duration-str started-at))
      (mapv extract-annotations responses))))

(s/fdef annotate-images
  :args (s/cat :service service? :images any? :opts (s/? (s/nilable map?))))

(defn annotate-image
  "Annotate `images` using the Google Cloud Vision `service`."
  [service image & [opts]]
  (first (annotate-images service [image] opts)))

(s/fdef annotate-image
  :args (s/cat :service service? :images any? :opts (s/? (s/nilable map?))))
