(ns burningswell.worker.addresses
  (:require [burningswell.db.addresses :as addresses]
            [burningswell.db.countries :as countries]
            [burningswell.db.regions :as regions]
            [burningswell.db.spots :as spots]
            [burningswell.rabbitmq.client :as rabbitmq]
            [clojure.spec.alpha :as s]
            [taoensso.timbre :as log]
            [com.stuartsierra.component :as component]
            [geo.core :as geo]
            [geocoder.google :as geocoder]
            [kithara.core :as k]
            [kithara.patterns.dead-letter-backoff
             :refer [with-durable-dead-letter-backoff]]
            [sqlingvo.core :refer [db?]]))

(s/def ::id int?)
(s/def ::db db?)
(s/def ::location geo/point?)

(s/def ::spot
  (s/keys :req-un [::id ::location]))

(defn country
  "Return the country for `location` or `result`."
  [db location result]
  (when-let [country (geocoder/country result)]
    (or (countries/closest db location)
        (countries/by-iso-3166-1-alpha-2 db (:iso-3166-1-alpha-2 country)))))

(s/fdef country
  :args (s/cat :db ::db :location ::location :result map?)
  :ret (s/nilable map?))

(defn region
  "Return the region for `location` or `result`."
  [db location result]
  (or (regions/closest db location)
      (regions/by-name db (geocoder/region result))))

(s/fdef region
  :args (s/cat :db ::db :location ::location :result map?)
  :ret (s/nilable map?))

(defn- resolve-result
  "Resolve the country and region of the geocoder result from `db`."
  [db location result]
  (if-let [country (country db location result)]
    (let [region (region db location result)]
      {:city (geocoder/city result)
       :country-id (:id country)
       :formatted (:formatted-address result)
       :location location
       :postal-code (geocoder/postal-code result)
       :region-id (:id region)
       :street-name (geocoder/street-name result)
       :street-number (geocoder/street-number result)})
    (log/warnf "Can't find country for address: %s."
               (:formatted-address result))))

(s/fdef resolve-result
  :args (s/cat :db ::db :location ::location :result map?)
  :ret (s/nilable map?))

(defn address-by-location
  "Return the address for `location`."
  [db location]
  (if-let [result (first (geocoder/geocode-location location))]
    (resolve-result db location result)
    (log/warnf "Can't geocode address for location %.3f, %.3f: No results."
               (geo/point-y location) (geo/point-x location))))

(s/fdef address-by-location
  :args (s/cat :db ::db :location ::location)
  :ret (s/nilable map?))

(defn geocode-address-by-location
  "Geocode and save the address of `location`."
  [db location]
  (when-let [address (address-by-location db location)]
    (if-let [existing (addresses/by-location db (:location address))]
      (addresses/update! db (merge existing address))
      (addresses/insert! db address))))

(s/fdef geocode-address-by-location
  :args (s/cat :db ::db :location ::location)
  :ret (s/nilable map?))

(defn log-geocode-success
  "Log a successful address geocode attempt."
  [spot address]
  (log/infof "Saved address \"%s\" (#%d) for spot \"%s\" (#%d)."
             (:formatted address) (:id address)
             (:name spot) (:id spot)))

(defn log-geocode-failure
  "Log a failing address geocode attempt."
  [spot]
  (log/warnf "Can't geocode address for location \"%s\" of spot \"%s\" (#%d)."
             (:location spot) (:name spot) (:id spot)))

(defn geocode-spot-address
  "Geocode the address of spots that have been created."
  [db spot]
  (when-let [spot (spots/by-id db (:id spot))]
    (if-let [address (geocode-address-by-location db (:location spot))]
      (log-geocode-success spot address)
      (log-geocode-failure spot))))

(s/fdef geocode-spot-address
  :args (s/cat :db ::db :spot ::spot)
  :ret (s/nilable map?))

(defmulti process-message
  "Handle messages that require geocoding of addresses."
  (fn [message] (-> message :routing-key keyword)))

(defmethod process-message :spots.created
  [{:keys [body env]}]
  (geocode-spot-address (:db env) body)
  {:status :ack})

(defmethod process-message :spots.updated
  [{:keys [body env]}]
  (when-not (addresses/by-location (:db env) (:location body))
    (geocode-spot-address (:db env) body))
  {:status :ack})

(defn worker
  "Return a new address worker."
  [config]
  (-> process-message
      (k/consumer {:as rabbitmq/read-edn :consumer-name "Address Geocoder"})
      (with-durable-dead-letter-backoff)
      (k/with-durable-queue "worker.addresses"
        {:exchange "api"
         :routing-keys ["spots.created" "spots.updated"]})
      (k/with-channel {:prefetch-count 1})
      (k/with-connection (rabbitmq/config (:broker config)))
      (k/with-env)
      (component/using [:db :topology])))
